grammar RubyMicro;
options {superClass=hotspots_x_ray.languages.InterruptibleParser;}

import RubySharedRulesLexer;

prog : global_expression*;

global_expression : global_variable_declaration
                  | singleton_definition
                  | endless_method
                  | function_definition
                  | fiber_creation
                  | bdd_description
                  | block_statement
                  | class_declaration
                  | singleton_class_declaration
                  | anything
                  ;

expression : singleton_definition
           | endless_method
           | function_definition
           | bdd_context
           | bdd_named_test
           | block_statement
           | class_declaration
           | singleton_class_declaration
           | anything
           ;

anything : ~(DEF | END | IF | UNLESS | WHILE | DO | FOR | CASE | BEGIN | MODULE);

// Special case to capture guard clauses like:
// new do
//   get :new
// end if parent_resource.actions.include?(:new)
end_with_guard: END IF;

global_variable_declaration: (global_variable_name | class_variable_name) '=';
global_variable_name: ID_GLOBAL;
class_variable_name: '@@' ID;

class_declaration : class_header crlf global_expression* END;
class_header : CLASS class_name class_base? SL_COMMENT?;
class_base : '<' ':'* (class_name (':'+ class_name)*);
class_name : ID;

singleton_class_declaration : CLASS '<<' SELF global_expression+ END;

// Endless methods are single-line method defintions introduced in Ruby 3.0.
// Example:
//   def raise_to_power(number, power) = number ** power
endless_method: endless_function_definition_header '=' anything+;

// Fibers are introduced in Ruby 3.0 for dealing with async tasks.
// We want to parse any top-level Fibers and treat the as named elements. That is,
// we make Fibers first-class citizens in the analysis (X-Ray and code health).
// Example:
//  f = Fiber.new do
//    puts "3: Entered fiber."
//    Fiber.yield
//  end
fiber_creation : function_name  CRLF*? '=' CRLF*?'Fiber.new' fiber_body;
fiber_body: DO expression+? (END | end_with_guard);

function_definition : function_definition_header function_definition_body END;

function_definition_header : DEF function_name function_definition_params
                           | DEF function_name;
endless_function_definition_header : DEF function_name function_definition_params;

singleton_definition : singleton_definition_header function_definition_body END;
singleton_definition_header : DEF SELF DOT function_name;

function_definition_body : expression*;

function_definition_params : LEFT_RBRACKET RIGHT_RBRACKET
                           | LEFT_RBRACKET function_definition_params_list RIGHT_RBRACKET
                           | function_definition_params_list
                           ;

function_definition_params_list : function_definition_param_id
                                | function_definition_params_list COMMA function_definition_param_id
                                ;

function_definition_param_id : single_param
                             | var_args
                             | keyword_args
                             | collected_keyword_args
                             | block_args;


single_param: function_argument_name function_definition_param_default_val?;
function_definition_param_default_val : '=' ~(COMMA | ')')+;// (anything | self_referenced_field);
self_referenced_field: SELF ('.' ID)+;

var_args: STAR function_argument_name;
keyword_args: function_argument_name ':' anything?;
collected_keyword_args: '**' function_argument_name;
block_args: '&' function_argument_name;

function_argument_name: ID;

function_name : ID_FUNCTION
              | ID '='?
              | LEFT_SBRACKET RIGHT_SBRACKET;

// BDD tests
//
bdd_description: DESCRIBE ~(DO | CRLF)+ DO bdd_description_body END;

bdd_description_body: expression*?;

bdd_context: CONTEXT bdd_context_name DO bdd_actual_test_body END;
bdd_context_name: LITERAL;

bdd_named_test: IT bdd_test_name DO bdd_actual_test_body END;
bdd_test_name: LITERAL;

bdd_actual_test_body: expression*?;

// Block elements
//

block_statement : crlf (IF | UNLESS | UNTIL | WHILE | DO | FOR | CASE | BEGIN) expression+ (END | end_with_guard)
                | returning_if
                | module_block_statement
                | do_block_statement
                | begin_block_statement
                | unless_single_line_statement
                | if_single_line_statement
                | while_single_line_statement;


module_block_statement : MODULE ID crlf global_expression+ END;
do_block_statement    : ~(DEF | END | IF | UNLESS | WHILE | FOR | CASE | BEGIN | MODULE | CRLF)+? DO expression+? (END | end_with_guard);
begin_block_statement : ~(CRLF)+? BEGIN expression+? (END | end_with_guard);

returning_if: '=' IF expression+ END;

// NOTE: just lookahead to crlf - do NOT consume it since we'll then miss a block_statement that follows immediately after!!
unless_single_line_statement : ~(DEF | END | IF | WHILE | DO | FOR | CASE | BEGIN | MODULE | CRLF)+? UNLESS ~(CRLF)+?;
if_single_line_statement     :  ~(DEF | END | UNLESS | WHILE | DO | FOR | CASE | BEGIN | MODULE | CRLF)+? IF ~(CRLF)+?;
while_single_line_statement  :  ~(DEF | END | UNLESS | DO | FOR | CASE | BEGIN | MODULE | CRLF)+? WHILE ~(CRLF)+?;

crlf : CRLF;
