grammar ECMAScriptNestedComplexity;

// This grammar detects islands of nested complexity.

method
:
	expression* EOF
;

expression : complexity
           | block_expression
           | anything;

complexity: if_clause
          | else_clause
          | switch_clause
          | for_in_loop
          | for_loop
          | while_loop
          | do_while_loop;

block_expression: OPENING_BRACE expression*? CLOSING_BRACE;

anything: ~(OPENING_BRACE | CLOSING_BRACE);

if_clause: IF some_condition (multi_line_conditional_expression | plain_line);
else_clause: ELSE (multi_line_conditional_expression | plain_line);

switch_clause: SWITCH some_condition multi_line_conditional_expression;

multi_line_conditional_expression: OPENING_BRACE expression*? CLOSING_BRACE;
plain_line: ~OPENING_BRACE ~(SEMICOLON)*? SEMICOLON;

for_in_loop: FOR LEFT_PAREN for_in_loop_part (IN | OF) for_loop_condition RIGHT_PAREN (multi_line_conditional_expression | plain_line);
for_in_loop_part:  ~(IN | SEMICOLON | OPENING_BRACE)+?;

for_loop: FOR LEFT_PAREN for_loop_part SEMICOLON for_loop_part SEMICOLON for_loop_condition RIGHT_PAREN (multi_line_conditional_expression | plain_line);
for_loop_part: ~(SEMICOLON)*?; // each part is optional
for_loop_condition: conditions*?;

while_loop: WHILE some_condition (multi_line_conditional_expression | plain_line);
do_while_loop: DO multi_line_conditional_expression WHILE some_condition;

some_condition: LEFT_PAREN conditions+? RIGHT_PAREN;
conditions : conditional_operator
           | ~(LEFT_PAREN | RIGHT_PAREN | SEMICOLON)
           | LEFT_PAREN conditions*? RIGHT_PAREN;

conditional_operator: CONDITIONAL_OPERATORS;

fragment ESCAPED : '\\\\' | '\\"';
LITERAL : '"' ( ESCAPED | ~('\n'|'\r') )*? '"';

fragment ESCAPED_SINGLE : '\\\\' | '\\\'';
SINGLE_LITERAL : '\'' (ESCAPED_SINGLE | ~('\n'|'\r') )*? '\'';

// Note: there's a nasty trap here that we fixed with the ~'>' rule.
// Without this rule, we would fail to close the block in the following kind of expression:
//    <Flex grow={1}>{systemId && <BadgeEntity systemId={systemId} size="sm" /> } </Flex>
fragment ESCAPED_SLASH : '\\/';
REGEX : '/' ~'>' (ESCAPED_SLASH | ~('\n'|'\r') )*? '/';

fragment ESCAPED_BACKTICK : '\\`';
BACKTICK_TEMPLATE : '`' ( ESCAPED_BACKTICK | ~('`') )*? '`';

LITERAL_CHAR : '\'' . '\'' -> skip;

IF: 'if';
ELSE: 'else';
FOR: 'for';
IN: 'in';
OF: 'of';
WHILE: 'while';
DO: 'do';
SWITCH: 'switch';
CONTINUE: 'continue';
RETURN: 'return';
THIS: 'this';

CONDITIONAL_OPERATORS: '&&' | '||';

ID : [a-zA-Z_][a-zA-Z0-9_]*;

SCOPER : '.';
SCOPED_NAME : ID (SCOPER ID)+;

LEFT_PAREN: '(';
RIGHT_PAREN: ')';
OPENING_BRACE: '{';
CLOSING_BRACE: '}';
COLON: ':';
SEMICOLON: ';';

Whitespace : [ \t]+ -> skip;

BlockComment: '/*' .*? '*/' -> skip;
LineComment: '//' ~[\r\n]* -> skip;

NEWLINE : '\r'? '\n' -> skip;

ANY_CHAR : .; // Put this lexer rule last to give it the lowest precedence