(ns codescene.features.util.maps
  (:require [camel-snake-kebab.core :as csk]
            [camel-snake-kebab.extras :as csk-extras]
            [clojure.string :as string]
            [clojure.walk :refer [walk]]
            [medley.core :refer [find-first]]))

(defn ->db-name
  "Convert the keyword or symbol to a DB compatible name:
  - snake case
  - no ? characters"
  [kw-or-sym]
  (-> (name kw-or-sym)
      csk/->snake_case
      (string/replace "?" "")
      keyword))

(defn- ->sym
  "Extract a symbol from the form.

  They symbol extracted will be:
  - the first symbol not in function name function call position.
  - the first keyword in function call position

  e.g. (->sym '(1 23 (inc {:a a}))) -> 'a'
       (->sym '(:x y)) -> 'x'
       (->sym '(inc y)) -> 'y'"
  [form]
  (walk ->sym
        #(if (coll? %) (find-first symbol? %) (when (symbol? %) %))
        (cond (and (seq? form) (symbol? (first form))) (rest form)
              (and (seq? form) (keyword? (first form))) (symbol (name (first form)))
              (map? form) (mapcat identity form)
              :else form)))

(defn rename-boolean-keys [m]
  (reduce-kv (fn [m k v] (assoc m (if (boolean? v) (keyword (namespace k) (str (name k) "?")) k) v))
             {}
             m))

(defn- -map-of
  "Helper function for map-of macros, allows varied key transformation.
  Symbol will be first symbol encountered in the form, not in function
  call name position."
  [key-xf forms]
  `(zipmap ~(vec (map (comp key-xf ->sym) forms)) ~(vec forms)))

(defmacro map-of-db
  "Creates map with symbol names as snake case keywords as keys and
   symbol values as values. Also removes ? from name

   Example: (map-of-db project-id name) => {:project_id project-id :name name}"
  [& syms]
  (-map-of ->db-name syms))

(defmacro map-of-kebab
  "Creates map with symbol names as kebab case keywords as keys and
   symbol values as values.

   Example: (map-of-kebab project_id name) => {:project-id project_id :name name}"
  [& syms]
  (-map-of (comp keyword csk/->kebab-case name) syms))

(defmacro map-of-kebab?
  "Creates map with symbol names as kebab case keywords as keys and
   symbol values as values. Adds ? to names when the value is boolean,
   beware of the interaction with nilable boolean keys!

   Example: (map-of-kebab project_id name) => {:project-id project_id :name name}"
  [& syms]
  `(rename-boolean-keys ~(-map-of (comp keyword csk/->kebab-case name) syms)))

(defmacro map-of
  "Creates map with symbol names as keywords as keys and
   symbol values as values.

   Example: (map-of id name) => {:id id :name name}"
  [& syms]
  (-map-of keyword syms))

(def ->db-keys
  (partial csk-extras/transform-keys ->db-name))

(def ->kebab
  "Just like ->kebab* but leave keys of boolean values alone"
  (partial csk-extras/transform-keys csk/->kebab-case))

(def ->kebab?
  "Also adds ? to names of boolean keys. Beware of situations where boolean
  keys can be also nil!"
  (comp rename-boolean-keys (partial csk-extras/transform-keys csk/->kebab-case)))

(def ->snake
  (partial csk-extras/transform-keys csk/->snake_case))

(defn- -group-by
  [[f & more] {:keys [acc-fn acc-init remove-keys] :as opt} acc it]
  (let [acc (or acc (first acc-init))
        acc-fn (or acc-fn #(conj (or %1 []) %2))
        k (f it)]
    (assoc acc
      k
      (if more
        (-group-by more (update opt :acc-init next) (get acc k) it)
        (acc-fn (get acc k)
                (apply dissoc it remove-keys))))))

(defn group-by*
  "Groups by a series of functions. Options:
  - acc-fn: (fn [acc x]) that is how leaf node will be accumulated,
            defaults for #(conj (or %1 []) %2), aka group-by
  - fn->comp: a fn that will return a comparator for a particular function,
  if one is found then a sorted-map is used.
  - dissoc-keys?: if set to true, then keys used to group by are removed from the
  leaves"
  [fns {:keys [acc-fn fn->comp dissoc-keys?]} coll]
  (let [remove-keys (if dissoc-keys? (filter ident? fns))
        acc-init (map #(if-some [cmp (get fn->comp %)] (sorted-map-by cmp) {}) fns)]
    (or (reduce (partial -group-by fns (map-of acc-fn acc-init remove-keys)) nil coll) {})))
