(ns codescene.features.pm-data.azure.azure-fetcher
  "Uses the Azure Rest API for fetching data.
   This includes combining and transforming data to internal pm-data format, and in some cases catching and reporting errors."
  (:require [taoensso.timbre :as log]
            [clojure.set :refer [rename-keys]]
            [codescene.cache.core :as cache]
            [codescene.features.pm-data.azure.azure-api :as azure-api]
            [codescene.features.util.log :as u.log]
            [codescene.pm-data.provider-common :as common]
            [com.climate.claypoole :as cp]
            [medley.core :as m]
            [clojure.string :as str]))

(defn- cost-from
  [fields cost-field]
  (or (get fields (keyword cost-field)) 0))

(defn fetch-projects
  "Fetches projects from the remote Azure API. Throws when the API calls fail."
  [api-client owner]
  (->> (azure-api/fetch-projects api-client owner)
       (map #(select-keys % [:name :id]))
       (map #(rename-keys % {:id :key}))))

(defn fetch-number-fields
  "Fetches number fields from the remote Azure API. Throws when the API calls fail."
  [api-client owner]
  (->> (azure-api/fetch-fields api-client owner)
       (filter #(#{"double" "integer"} (:type %)))
       (map #(select-keys % [:name :referenceName]))
       (map #(rename-keys % {:referenceName :key}))))

(defn fetch-board-columns
  "Fetches board columnnsfrom the remote Azure API. Throws when the API calls fail."
  [api-client owner]
  (->> (azure-api/fetch-board-columns api-client owner)
       (map :name)
       (map common/->name-and-key)))

(defn- fetch-work-item-types*
  "Fetches work item types from the remote Azure API. Order of work items is not preserved. Throws when the API calls fail."
  [api-client owner projects]
  (when (empty? projects)
    ;; TODO: check if this always holds and what the consequences are when throwing this exception
    (throw (ex-info "there should be at least 1 project" {:owner owner})))

  (u.log/log-time
   (let [pool-size (min (count projects) 50)]
     (->> projects
          ;; NOTE: the `pool-size` cannot be 0 otherwise this whole thing blocks forever (this is)
          (cp/upmap pool-size (fn [p] (azure-api/fetch-work-item-types api-client owner p)))
          (apply concat)
          vec))
   :info
   "PM Data: Fetching Azure DevOps work-item-types for %s projects" (count projects)))

(defn fetch-work-item-types
  "Fetches work item types from the remote Azure API. Order of work items is not preserved. Throws when the API calls fail."
  [api-client owner projects]
  (->> (fetch-work-item-types* api-client owner projects)
       (map :name)
       distinct
       (mapv common/->name-and-key)))

(defn fetch-work-item-states
  "Fetches work item states from the remote Azure API. Order of work items is not preserved. Throws when the API calls fail."
  [api-client owner projects]
  ;; NOTE: this is cached (see the end of this ns) so the duplication in `fetch-work-item-types` is smaller problem
  ;; - i.e.  we don't actually fetch work items twice)
  (->> (fetch-work-item-types* api-client owner projects)
       (mapcat :states)
       (map :name)
       distinct
       (mapv common/->name-and-key)))

(defn- ->transitions
  [revisions]
  (->> revisions
       (map :fields)
       (map #(update % :System.ChangedDate common/drop-ms-from-time-string))
       (map (juxt :System.State :System.ChangedDate))
       common/keep-first-transitions
       (into [])))

(defn revisions->issue
  [provider-def [id revisions]]
  (let [{:keys [fields]} (last revisions)
        {title :System.Title created :System.CreatedDate updated :System.ChangedDate status :System.State work-type :System.WorkItemType} fields
        {:keys [cost-field]} provider-def
        transitions (->transitions revisions)]
    (when (str/blank? title)
      (log/info "Task " id " has a blank System.Title, available fields:" fields))
    {:id (str id)
     :task-name title
     :created  (common/drop-ms-from-time-string created)
     :closed nil ;; Azure issues have no closed property(?)
     :updated updated
     :status status
     :cost (cost-from fields cost-field)
     :work-types [work-type]
     :transitions transitions}))

(defn group-by-work-item
  [coll]
  (->> coll
       (sort-by (comp :System.ChangedDate :fields))
       (group-by :work-item-id)))

(defn fetch-work-item-revisions
  "Fetches work item revisions from the remote Azure API."
  [since api-client {:keys [owner project] :as _repo-parts} provider-def]
  (let [{:keys [cost-field]} provider-def
        search-options {:owner owner
                        :project project
                        :since since}
        result-options {:fields (concat ["System.Title" "System.WorkItemType" "System.CreatedDate" "System.ChangedDate" "System.State"]
                                        (when-not (empty? cost-field) [cost-field]))}]
    (log/infof "Fetch work item revisions from Azure since %s" (or since "-"))
    (let [revisions (->> (azure-api/fetch-reporting-work-item-revisions api-client search-options result-options)
                         (mapv (fn [{:keys [id rev] :as revision}]
                                 (assoc revision
                                        :id (str id "-" rev)
                                        :work-item-id id))))]
      (log/infof "Fetched %d work item revisions from Azure" (count revisions))
      revisions)))

(defn- link-type-is? [link link-type]
  (= (get-in link [:attributes :linkType]) link-type))

(defn- link->response [link]
  (let [{:keys [attributes]} link
        {:keys [sourceId targetId changedDate]} attributes]
    {:id     (str targetId)
     :parent (str sourceId)
     :updated changedDate}))

(defn fetch-links
  [since api-client {:keys [owner project] :as _repo-parts} _provider-def]
  (let [search-options {:owner owner
                        :project project
                        :link-types ["System.LinkTypes.Hierarchy"]
                        :since since}]
    (log/infof "Fetch links from Azure since %s" (or since "-"))
    (let [links (->> (azure-api/fetch-reporting-work-item-links api-client search-options)
                     (filter #(link-type-is? % "System.LinkTypes.Hierarchy-Forward"))
                     (mapv #(link->response %)))]
      (log/infof "Fetched %d links from Azure" (count links))
      links)))

(defn- pr->result
  [{:keys [pullRequestId completionOptions commits] :as _pr}]
  {:id (str pullRequestId)
   :text (:mergeCommitMessage completionOptions)
   :commits (map :commitId commits)})

(defn fetch-pull-request
  "Fetches pull-requests from the remote Azure API. Throws when the API calls fail."
  [api-client repo-parts pr]
  (log/infof "Fetch pull request %s from Azure" pr)
  (->> (azure-api/fetch-pull-request api-client repo-parts pr)
       pr->result))

(defn fetch-pull-requests
  "Fetches completed pull-request ids from the remote Azure API. Throws when the API calls fail."
  ([api-client repo-parts]
   (fetch-pull-requests api-client repo-parts nil))
  ([api-client repo-parts n]
   (log/infof "Fetch pull requests")
   (let [search-options (m/assoc-some {:status "completed"}
                                      :count n)
         prs (->> (azure-api/fetch-pull-requests api-client repo-parts search-options)
                  (mapv :pullRequestId))]
     (log/infof "Fetched %d pull requests from Azure" (count prs))
     prs)))

(cache/memo-scoped #'fetch-work-item-types*)
(cache/memo-scoped #'fetch-board-columns)
(cache/memo-scoped #'fetch-number-fields)
(cache/memo-scoped #'fetch-projects)

(comment
  (def api-client ["" (System/getenv "AZURE_TOKEN")])
  (def owner "empear")
  (def project "SmartHotel360")
  (def project "dcd6233e-c108-4b8a-bb00-5f39a1a978f0") ; project with spaces
  (def repo "PublicWeb")
  (def repo-url "git@ssh.dev.azure.com:v3/empear/SmartHotel360/PublicWeb")
  (def repo-url "git@ssh.dev.azure.com:v3/empear/Project%20With%20Spaces/Repo%20With%20Spaces")
  (def repo-parts (codescene.url.url-utils/azure-url->parts repo-url))
  (fetch-pull-requests api-client repo-parts)
  (fetch-work-item-revisions (evolutionary-metrics.trends.dates/string->date "2024-01-01")
                             api-client repo-parts
                             {:cost-unit "points"
                              :cost-field "Microsoft.VSTS.Scheduling.Effort"})
  (fetch-links nil api-client repo-parts {})
  (fetch-work-item-types api-client owner ["650a76c2-d1f4-4085-a6b8-d87bf4e5a6bd"])
  (fetch-work-item-states api-client owner ["650a76c2-d1f4-4085-a6b8-d87bf4e5a6bd"])
  (fetch-board-columns api-client owner)
  (fetch-number-fields api-client owner)
  (fetch-projects api-client owner)

  (cache/with-scoped-cache
    (let [futures (mapv (fn [_] (future (fetch-pull-requests api-client repo-parts))) (range 10))]
      (mapv deref futures)))
  )
