(ns codescene.features.delta.pr-status
  (:require [clojure.java.jdbc :as jdbc]
            [clojure.string :as str]
            [codescene.cache.core :as cache]
            [codescene.features.util.maps :as maps :refer [group-by* map-of-db]]
            [codescene.features.delta.integration.integration-db :as integration-db]
            [codescene.features.delta.code-health.ch-config :as ch]
            [codescene.features.config.properties :as p]
            [hugsql.core :as hugsql]))

(hugsql/def-db-fns "codescene/features/delta/pr_status.sql")

(defn get-project-open-prs [tx project-id]
  (->> (get-open-prs tx (map-of-db project-id))
       (mapv #(dissoc % :project_id))
       (group-by* [:repo_id :pr_id]
                  {:dissoc-keys? true :acc-fn #(:sha %2)})))

(defn upsert-open-pr [db-spec {:keys [project-id repo-id pr-id sha]}]
  (let [data (map-of-db project-id (str/lower-case repo-id) pr-id sha)]
    (jdbc/with-db-transaction [tx db-spec]
      (if (get-open-pr tx data)
        (update-open-pr! tx data)
        (insert-open-pr! tx data)))))

(defn delete-open-pr [tx project-id repo-id pr-id]
  (delete-open-pr! tx (map-of-db project-id (str/lower-case repo-id) pr-id)))

(defn has-open-pr-support
  {cache/conf (cache/inf)}
  [tx project-id]
  ;; cache if supported since it cannot become unsupported
  (or (-> (integration-db/get-config tx project-id (constantly [ch/open-pr-support]))
          :data
          :open-pr-support)
      (cache/do-not-cache false)))

(cache/memo #'has-open-pr-support)

(defn mark-open-pr-support [tx project-id provider-id]
  (integration-db/update-configuration
    tx
    (p/config-set {:project-id project-id
                   :provider-id provider-id}
                  {:open-pr-support true}
                  [ch/open-pr-support]
                  :live)))