(ns codescene.features.chat.service.llm.openai
  "OpenAI and OpenAI-compatible API provider (Azure, custom endpoints)."
  (:require [clojure.data.json :as json]
            [codescene.features.chat.service.llm.http :as http]
            [codescene.features.chat.service.llm.provider :as provider]
            [taoensso.timbre :as log]))

(defn- status-code->user-hint
  "Returns a user-friendly hint for common HTTP status codes."
  [status]
  (case status
    401 "Please verify your API key is correct and active."
    429 "Rate limit exceeded. Please check your OpenAI billing and usage limits at https://platform.openai.com/account/billing"
    500 "OpenAI is experiencing issues. Please try again later."
    502 "OpenAI is temporarily unavailable. Please try again later."
    503 "OpenAI service is overloaded. Please try again later."
    nil))

(defn- normalize-usage
  "Normalizes OpenAI usage format to {:prompt-tokens N :completion-tokens N}."
  [usage]
  (when usage
    {:prompt-tokens (:prompt_tokens usage)
     :completion-tokens (:completion_tokens usage)}))

(defn- parse-response
  "Parses an OpenAI API response into our standard format.
   Extracts content and usage stats on success, error message on failure.
   Normalizes usage to {:prompt-tokens N :completion-tokens N} format."
  [response]
  (if (= 200 (:status response))
    {:success true
     :content (get-in response [:body :choices 0 :message :content])
     :usage (normalize-usage (get-in response [:body :usage]))}
    (let [status (:status response)
          api-error (http/extract-error response nil)
          user-hint (status-code->user-hint status)
          error-msg (cond
                      (and api-error user-hint) (str api-error " " user-hint)
                      api-error api-error
                      user-hint (str "API error: " status ". " user-hint)
                      :else (str "API error: " status))]
      {:success false
       :error error-msg})))

(defn- send-request
  "Sends a request to an OpenAI-compatible API.
   
   Arguments:
   - opts: Map with :url, :api-key, :model, :messages, :http-opts"
  [{:keys [url api-key model messages http-opts]}]
  (http/post url
             (merge {:headers {"Authorization" (str "Bearer " api-key)
                               "Content-Type" "application/json"}
                     :body (json/write-str {:model model
                                            :messages (http/format-messages messages)})
                     :as :json}
                    http-opts)))

(defrecord OpenAIProvider []
  provider/LLMProvider
  
  (call [_ {:keys [base-url chat-endpoint api-key model http-opts]} messages]
    (let [url (str base-url chat-endpoint)]
      (log/debugf "Calling OpenAI-compatible API at %s" url)
      (let [response (send-request {:url url
                                    :api-key api-key
                                    :model model
                                    :messages messages
                                    :http-opts http-opts})]
        (if (http/request-failed? response)
          (http/format-request-error response)
          (parse-response response))))))
