(ns codescene.features.chat.service.llm.http
  "HTTP utilities for LLM provider API calls."
  (:require [clj-http.client :as http]))

(defn format-messages
  "Converts internal message format to API-compatible format.
   Transforms `{:role :user :content \"text\"}` to `{:role \"user\" :content \"text\"}`."
  [messages]
  (mapv (fn [{:keys [role content]}]
          {:role (name role) :content content})
        messages))

(defn post
  "Executes an HTTP POST with exception handling.
   Returns the response map on success, or {:error-exception e} on failure."
  [url opts]
  (try
    (http/post url (assoc opts :throw-exceptions false))
    (catch Exception e
      {:error-exception e})))

(defn extract-error
  "Extracts error message from an API response body.
   Tries common error paths, falls back to default-msg."
  [response default-msg]
  (or (get-in response [:body :error :message])
      (get-in response [:body :error])
      default-msg))

(defn request-failed?
  "Returns true if the response contains an exception from a failed request."
  [response]
  (boolean (:error-exception response)))

(defn format-request-error
  "Formats an exception from a failed HTTP request into an error response."
  [response]
  {:success false 
   :error (str "Request failed: " (.getMessage (:error-exception response)))})
