(ns codescene.features.api.safeguards
  "REST API endpoints for Code Health safeguards.
   Provides insights and outcomes for Code Health quality gates from various sources.
   Currently supports PR-based safeguards, extensible to MCP and IDE sources."
  (:require [codescene.features.api.core :as api-core]
            [codescene.features.components.analysis :as analysis]
            [codescene.features.util.api :as api-utils]
            [codescene.util.date :as date-utils]
            [codescene.analysis.paths :as paths]
            [codescene.util.json :as json]
            [codescene.delta.statistics.trends :as trends]
            [clj-time.core :as t]
            [evolutionary-metrics.trends.dates :as dates]))

(defn- trend-in-date-range?
  "Check if a monthly trend falls within the date range"
  [{:keys [date]} from-date to-date]
  (when-let [trend-date (trends/yy-mm->date date)]
    (and (not (t/before? trend-date from-date))
         (not (t/after? trend-date to-date)))))

(defn- interpret-trend
  "Interpret the trend data to provide actionable insights.
   Positive net-change means issues are increasing (bad).
   Negative or zero net-change means issues are stable or decreasing (good)."
  [{:keys [total-detected-violations total-detected-violations-fixed
           total-violations-ignored total-negative-findings
           total-ignored-findings total-suppressed-findings]
    :or {total-detected-violations 0
         total-detected-violations-fixed 0
         total-violations-ignored 0
         total-negative-findings 0
         total-ignored-findings 0
         total-suppressed-findings 0}}]
  (let [resolved (+ total-detected-violations-fixed total-violations-ignored)
        net-change (- total-detected-violations resolved)
        addressed-findings (+ total-ignored-findings total-suppressed-findings)]
    {:net-change net-change
     :status (cond
               (zero? total-detected-violations) :no-issues
               (neg? net-change) :improving
               (zero? net-change) :stable
               :else :degrading)
     :findings-identified total-negative-findings
     :findings-addressed addressed-findings
     :findings-net-change (- total-negative-findings addressed-findings)}))

(defn- summarize-trends
  "Summarize multiple monthly trends into aggregate statistics"
  [trends]
  (let [summaries (map :summary trends)
        sum-field (fn [field]
                    (reduce + 0 (map #(get % field 0) summaries)))]
    {:total-delta-analyses (sum-field :total-delta-analyses)
     :total-detected-violations (sum-field :total-detected-violations)
     :total-detected-violations-fixed (sum-field :total-detected-violations-fixed)
     :total-violations-ignored (sum-field :total-violations-ignored)
     :total-negative-findings (sum-field :total-negative-findings)
     :total-ignored-findings (sum-field :total-ignored-findings)
     :total-suppressed-findings (sum-field :total-suppressed-findings)}))

(defn- pr-trends->insights
  "Transform PR statistics trends into insights response"
  [trends meta-info]
  (let [summary (summarize-trends trends)
        interpretation (interpret-trend summary)]
    {:meta meta-info
     :summary {:total-pr-analyses (:total-delta-analyses summary)
               :findings-identified (:total-negative-findings summary)
               :findings-fixed (:total-detected-violations-fixed summary)
               :findings-ignored (:total-ignored-findings summary)
               :findings-suppressed (:total-suppressed-findings summary)
               :violations-detected (:total-detected-violations summary)
               :violations-resolved (+ (:total-detected-violations-fixed summary)
                                       (:total-violations-ignored summary))}
     :interpretation interpretation
     :monthly-trends (mapv (fn [{:keys [date summary]}]
                             {:month date
                              :statistics summary
                              :interpretation (interpret-trend summary)})
                           trends)}))

(defn- pr-stat-to-outcome
  "Convert a PR statistics entry to an outcome"
  [pr-stat]
  (let [{:keys [external-review-id last-pr-analysis-date repo authors
                total-reported-degradations total-new-files-with-low-health
                fixed-code-health-issues ignored-code-health-issues
                total-negative-findings total-positive-findings
                ignored-findings suppressed-findings]} pr-stat
        total-issues (+ total-reported-degradations total-new-files-with-low-health)
        resolved (+ fixed-code-health-issues ignored-code-health-issues)
        net-change (- total-issues resolved)]
    {:pr-id external-review-id
     :analysis-date last-pr-analysis-date
     :repository repo
     :authors authors
     :statistics {:violations-detected total-issues
                  :violations-fixed fixed-code-health-issues
                  :violations-ignored ignored-code-health-issues
                  :negative-findings total-negative-findings
                  :positive-findings total-positive-findings
                  :findings-ignored ignored-findings
                  :findings-suppressed suppressed-findings}
     :interpretation {:net-change net-change
                      :status (cond
                                (zero? total-issues) :no-issues
                                (<= net-change 0) :resolved
                                :else :unresolved)}}))

(defn- pr-in-date-range?
  "Check if a PR's last analysis date falls within the date range"
  [{:keys [last-pr-analysis-date]} from-str to-str]
  (when last-pr-analysis-date
    (try
      (let [pr-date-str (subs last-pr-analysis-date 0 10)] ;; Extract YYYY-MM-DD from ISO string
        (and (>= (compare pr-date-str from-str) 0)
             (<= (compare pr-date-str to-str) 0)))
      (catch Exception _
        false))))

(defn- empty-insights-response
  [meta-info]
  {:meta meta-info
   :summary {:total-pr-analyses 0
             :findings-identified 0
             :findings-fixed 0
             :findings-ignored 0
             :findings-suppressed 0
             :violations-detected 0
             :violations-resolved 0}
   :interpretation {:net-change 0
                    :status :no-issues
                    :findings-identified 0
                    :findings-addressed 0
                    :findings-net-change 0}
   :monthly-trends []})

(defn- insights-from-analysis
  [{:keys [analysis-component project-id analysis-id from-date to-date meta-info]}]
  (let [path-fn (analysis/local-analysis-result-path analysis-component project-id analysis-id)
        trends-file (path-fn paths/delta-trend-results-json)
        trends (let [data (json/read-when-exists trends-file)]
                 (if (seq data) data []))
        filtered-trends (filter #(trend-in-date-range? % from-date to-date) trends)]
    (pr-trends->insights filtered-trends meta-info)))

(defn insights
  "Get aggregated insights for code health quality gates for a project within a date range.
   This reads the PR statistics trends data (delta_trend_results.json) from the latest analysis."
  [system {:keys [project-id from to]}]
  (let [generated-at (t/now)
        from-date-str (date-utils/get-date-string from (t/minus generated-at (t/months 1)))
        to-date-str (date-utils/get-date-string to generated-at)
        from-date (dates/string->date from-date-str)
        to-date (dates/string->date to-date-str)
        analysis-component (api-core/api-analyses system)
        latest-analysis (analysis/latest-successful-analysis analysis-component project-id)
        meta-info {:project-id project-id
                   :from from-date-str
                   :to to-date-str
                   :generated-at generated-at}]
    (api-utils/ok
      (if latest-analysis
        (insights-from-analysis {:analysis-component analysis-component
                                 :project-id project-id
                                 :analysis-id (:id latest-analysis)
                                 :from-date from-date
                                 :to-date to-date
                                 :meta-info (assoc meta-info :analysis-id (:id latest-analysis))})
        (empty-insights-response meta-info)))))

(defn- outcomes-from-analysis
  [{:keys [analysis-component project-id analysis-id from-date-str to-date-str]}]
  (let [path-fn (analysis/local-analysis-result-path analysis-component project-id analysis-id)
        pr-stats-file (path-fn paths/delta-detailed-pr-statistics-result-json)
        pr-stats (let [data (json/read-when-exists pr-stats-file)]
                   (if (seq data) data []))
        filtered-prs (filter #(pr-in-date-range? % from-date-str to-date-str) pr-stats)]
    (mapv pr-stat-to-outcome filtered-prs)))

(defn outcomes
  "Get individual PR outcomes for code health quality gates for a project within a date range.
   This reads the detailed PR statistics (delta_detailed_pr_statistics_result.json) from the latest analysis."
  [system {:keys [project-id from to page page-size]}]
  (let [from-date-str (date-utils/get-date-string from (t/minus (t/now) (t/months 1)))
        to-date-str (date-utils/get-date-string to (t/now))
        analysis-component (api-core/api-analyses system)
        latest-analysis (analysis/latest-successful-analysis analysis-component project-id)
        results (if latest-analysis
                  (outcomes-from-analysis {:analysis-component analysis-component
                                           :project-id project-id
                                           :analysis-id (:id latest-analysis)
                                           :from-date-str from-date-str
                                           :to-date-str to-date-str})
                  [])]
    (api-utils/paginated-list-result results
                                     {:page page
                                      :page-limit page-size
                                      :list-kw :outcomes
                                      :processing-fn identity})))
