(ns codescene.features.util.string
  "Handy utility functions for dealing with strings.
  Most of the time you should probably favor using `clojure.string` directly.")

(defn truncate
  "Truncate a string with suffix (ellipsis by default) only if it is
   longer than specified length."

  ([^String string length suffix]
   {:pre [(>= length 0)]}
   (when string
     (let [suffix-len (count suffix)
           string-len (.length string)]
       (if (<= string-len length)
         string
         (str (subs string 0 (max 0 (- length suffix-len))) suffix)))))

  ([^String string length]
   (truncate string length "...")))

(defn hide
  "Truncate a 'string' to be max the given 'length' including the ellipsis,
   but hide parts of it even if it's shorter than 'length', shows at most half characters."

  [^String string length]
  {:pre [(>= length 0)]}
  (when string
    (let [size (min length (quot (.length string) 2))
          half (quot size 2)]
      (str (subs string 0 (- size half))
           "..."
           (subs string (- (.length string) half))))))

(defn safe-match-some?
  "Returns true if string matches regex at any position. nil or empty regex is considered to never
   match, nil string is considered to never match, even if regex is nil. Invalid regex never matches."
  [regex-str s]
  (try
    (boolean (and s (seq regex-str) (re-find (re-pattern regex-str) s)))
    (catch Exception _ false)))

(defn declob
  "Turn a clob into a String"
  [clob]
  (condp instance? clob
    String clob
    java.sql.Clob (try
                    (slurp (.getCharacterStream ^java.sql.Clob clob))
                    (catch Exception e
                      ;; https://empear.slack.com/archives/C5LJXDATH/p1759123748312149
                      ;; Work around strange "The object is already closed" errors for objects where the value is in fact accessible:
                      (if (System/getProperty "codescene.onprem.test-suite-running")
                        (let [field (.getDeclaredField (Class/forName "org.h2.jdbc.JdbcLob") "value")]
                          (.setAccessible field true)
                          (.getString (.get field clob)))
                        (throw e))))
    nil))