(ns codescene.features.util.api
  (:require [clojure.spec.alpha :as s]
            [clojure.walk :as w]
            [clojure.string :as string]
            [codescene.presentation.display :as display]
            [codescene.features.util.item-filter :as item-filter]
            [codescene.features.util.log :as log-util]
            [medley.core :as m])
  (:import clojure.lang.ExceptionInfo))

(def ^:const API_V2 "v2")
;; in om-prem api-route is used to differentiate between ui and api
;; in cloud we have api.codescene.io which make the /api to look like a duplicate
;; we use api-root env var for that or default-api-root when env var is not set
(def ^:const default-api-root "/api")

;; we define versions here, in an ideal world we should have not more then 2 versions, where previous one is deprecated
(def ^:const api-versions [{:version API_V2 :deprecated false :latest true :order 1}])

(def ^:const default-page-limit 100)

(defn get-api-version
  "get api version based on version string(v1, v2, etc), nil if not found"
  [version]
  (m/find-first #(= version (:version %)) api-versions))

(defn at-least-version?
  "check if order of current-version is greater or equal then given version
  versions are identified by their string label: v1, v2, etc
  if no current-version found the result is false"
  [current-version version]
  (let [current-order (:order (get-api-version current-version))]
    (and current-order (>= current-order (:order (get-api-version version))))))

(defn up-to-version?
  "check if order of current-version is smaller or equal then given version
  versions are identified by their string label: v1, v2, etc
  if no current-version found the result is false"
  [current-version version]
  (let [current-order (:order (get-api-version current-version))]
    (and current-order (<= current-order (:order (get-api-version version))))))

(defn with-base-url
  ([uri]
   (with-base-url default-api-root uri))
  ([api-prefix uri]
   (let [last-version (-> (m/find-first :latest api-versions) :version)]
     (with-base-url api-prefix last-version uri)))
  ([api-prefix api-version uri]
   (format "%s/%s%s" api-prefix api-version uri)))

(defn base-url [api-prefix]
  (with-base-url api-prefix ""))

(defn project-url-prefix
  [api-prefix] (with-base-url api-prefix "/projects/"))

(defn user-url-prefix []
  (with-base-url default-api-root "/users/"))

(defn role-url-prefix []
  (with-base-url default-api-root "/roles/"))

(defn authentication-provider-url-prefix []
  (with-base-url default-api-root "/authentication-providers/"))

(defn developer-settings-url-prefix
  [api-prefix] (with-base-url api-prefix "/developer-settings/"))

(defn code-coverage-url-prefix
  [api-prefix] (with-base-url api-prefix "/code-coverage/"))

(defn group-url-prefix []
  (with-base-url default-api-root "/groups/"))

(defn json-safe-keys
  "replace in keyword all -(dash) with _(underscore)"
  [m]
  (let [replace-fn (fn [m k v] (assoc m (if (keyword? k) (keyword (string/replace (name k) \- \_)) k) v))
        ;; only apply to maps
        postwalk-fn (fn [x] (if (map? x) (reduce-kv replace-fn {} x) x))]
    (w/postwalk postwalk-fn m)))

(defn rename-and-format-to-int
  [data map-of-keys-to-rename list-of-keys-to-format]
  (let [s (set list-of-keys-to-format)]
    (reduce-kv
      (fn [m k v] (assoc m (map-of-keys-to-rename k k)
                           (cond-> v
                             (s k) display/->maybe-int)))
      {}
      data)))

(defn- json-safe-response
  [status body headers]
  {:status  status
   :headers (merge {"Content-Type" "application/json"}
                   headers)
   :body    (json-safe-keys body)})

(defn- mask-secrets
  "look for all strings in the body and mask secrets, convert all keywords to string"
  [body]
  (cond
    (keyword? body) (name body)
    (string? body) (log-util/mask-secrets-in-log body)
    (map? body) (reduce-kv (fn [m k v] (assoc m k (mask-secrets v))) {} body)
    (coll? body) (map #(mask-secrets %) body)
    :else body))

(defn response
  [status body]
  (json-safe-response status {:error (mask-secrets body)} {}))

(defn not-found
  [body]
  (json-safe-response 404 {:error (mask-secrets body)} {}))

(defn bad-request
  [body]
  (json-safe-response 400 {:error (mask-secrets body)} {}))

(defn internal-server-error
  [body]
  (json-safe-response 500 {:error (mask-secrets body)} {}))

(defn unauthorized
  [body]
  (json-safe-response 401 {:error (mask-secrets body)} {}))

(defn forbidden
  [body]
  (json-safe-response 403 {:error (mask-secrets body)} {}))

(defn ok
  ([body]
   (ok body {}))
  ([body headers]
   (json-safe-response 200 body headers)))

(defn created
  [body headers]
  (json-safe-response 201 body headers))

(defn accepted
  [body headers]
  (json-safe-response 202 body headers))

(defn page-with
  "split data in pages based on page-limit and return a page with each element processed by processing-fn"
  [data page page-limit]
  (->> data
       (drop (* (- page 1) page-limit))
       (take page-limit)))

(defn paginated-list-result*
  "Returns json list result for data paginated and processed. Does not include a Ring response-shaped wrapper object."
  [data {:keys [page page-limit filter-str list-kw processing-fn extra-map]
         :or {page-limit default-page-limit extra-map {}}}]
  {:pre [processing-fn]}
  (let [filtered-data (->> data
                           (map processing-fn)
                           (filter (item-filter/parse-filter filter-str)))
        numberOfElements (count filtered-data)
        max_pages (int (Math/ceil (double (/ numberOfElements page-limit))))]
    (if (> numberOfElements 0)
      (if (and (> page 0) (<= page max_pages))
        (merge {:page page
                :max_pages max_pages
                list-kw (page-with filtered-data page page-limit)}
               extra-map)
        (throw (ex-info (format "There's no data for the given page: %d" page)
                        {:page page
                         :kind :paginated-list-result})))
      {:page page :max_pages max_pages list-kw []})))

(defn paginated-list-result
  "get json list result for data paginated and processed, wrapped in Ring response format."
  [data {:keys [page page-limit filter-str list-kw processing-fn extra-map] :or {page-limit default-page-limit extra-map {}}} ]
  {:pre [processing-fn]}
  (try
    (ok (paginated-list-result* data {:page page
                                      :page-limit page-limit
                                      :filter-str filter-str
                                      :list-kw list-kw
                                      :processing-fn processing-fn
                                      :extra-map extra-map}))
    (catch ExceptionInfo e
      (if (-> e ex-data :kind (= :paginated-list-result))
        (not-found (ex-message e))
        (throw e)))))

(defn handle-validation-error [_ data _]
  (let [parameters (get-in data [:problems ::s/value])
        explanation (->> (get-in data [:problems ::s/problems])
                         (mapv (fn [{:keys [path pred val]}]
                                 (format "At path %s, value %s doesn't match predicate %s" path (pr-str val) pred)))
                         (string/join \newline))]
    (bad-request (format "Invalid values for request parameters: %s\n%s" parameters explanation))))
