(ns codescene.features.reports.graphics.doughnut
  (:import [java.awt Color BasicStroke]))

(defn- awt-color [color]
  (let [[r g b] color]
    (Color. r g b)))

(defn- calculate-positions
  "Calculates startAngle:s for the drawArc function, i.e. the position to start each extent.
  The function returns an array of angles, one for each extent plus one for the end position, starting at initial-pos.
  Angles are interpreted such that 0 degrees is at the 3 o'clock position and 90 degrees at the 12 o'clock position. "
  [initial-pos angular-extents]
  (loop [f initial-pos
         c angular-extents
         ret [initial-pos]]
    (if (seq c)
      (let [next-pos (+ f (first c))]
        (recur next-pos (rest c) (conj ret next-pos)))
      ret)))

(comment
  (calculate-positions 90  [216N 144N])
  (calculate-positions 90 [-60 -120 -180]))


(defn- calculate-angular-extents
  "Calculates arcAngle:s for the drawArc function: i.e. the length of the arc in degrees.
  Assumes that the values should fill up the total 360 degrees.
  Rounds the angles to integer values."
  [values]
  (let [total-value (apply + values)
        ratios (map #(/ % total-value) values)
        lengths (map #(* % 360) ratios)
        rounded-lengths (map #(Math/round (double %)) lengths)]
    rounded-lengths))

(comment
  (calculate-angular-extents  [0 6 4])
  (calculate-angular-extents nil))

(defn draw-doughnut
  "Draws a doughnut diagram with any number of segments. Can also draw text in the donut.
  The g2d, width and height parameters depend on the surrounding boundng boxes.
  donut-otions:
   :values is an array of values, giving the relative size of each segment.
   :colors is an array of colors, one per value
   :end-cap can be one of  BasicStroke/CAP_BUTT BasicStroke/CAP_SQUARE BasicStroke/CAP_ROUND
   :stroke-width is the width of the stroke
  text-options - note that text is fidgety and needs a lot of manual testing and nudging to look good.
  The selection of font is also limited.
   :text is the text to be drawn
   :text-color is the color for the text
   :font-style is for example java.awt.Font/PLAIN java.awt.Font/BOLD java.awt.Font/ITALIC
   :font-size is the font size :-)"
  [g2d width height {:keys [doughnut-options text-options]}]
  (let [{:keys [values colors end-cap stroke-width]} doughnut-options
        extents (calculate-angular-extents values)
        clockwise-extents (map #(* -1 %) extents)
        positions (calculate-positions 90 clockwise-extents)
        segments (map vector colors positions clockwise-extents)
        strokeWidth (float stroke-width)
        size (min width height)
        left (max 0 (/ (- width height) 2))
        top (max 0 (/ (- height width) 2))]
    (when (some? text-options)
      (let [{:keys [font-style font-size]} text-options]
        (doto g2d
          (.setFont (java.awt.Font. "helvetica" font-style font-size))))
      (let [metrics (.getFontMetrics g2d)
            {:keys [text text-color nudge-text]} text-options
            string-width (.stringWidth metrics text)]
        (doto g2d
          (.setColor (awt-color text-color))
          (.drawString text (float (/ (- width string-width) 2)) (float (+ (/ height 2) nudge-text))))))
    (.setStroke g2d (BasicStroke. strokeWidth end-cap BasicStroke/JOIN_ROUND))
    (doseq [[color position extent] segments]
      (when (< extent 0)
        (doto g2d
          (.setColor (awt-color color))
          (.drawArc (int left) (int top) (int size) (int size) position (+ 3 extent)))))
    g2d
    ))