(ns codescene.features.delta.suppress
  (:require
    [clojure.spec.alpha :as s]
    [codescene.delta.statistics.specs :as statistics-specs]
    [codescene.features.util.maps :refer [->db-keys group-by* ->kebab map-of-db]]
    [codescene.features.util.coll :refer [do-diff]]
    [codescene.specs :as specs]
    [hugsql.core :as hugsql]
    [taoensso.timbre :as log])
  (:import (java.util Date)))

(hugsql/def-db-fns "codescene/features/delta/delta_suppress.sql")

(s/def ::fn-name (s/nilable (specs/string-n 255)))
(s/def ::suppressed-by (s/nilable (specs/string-n 255)))
(s/def ::file-name (specs/string-n 1024))
(s/def ::reason (s/nilable (specs/string-n 4096)))
(s/def ::biomarker (specs/string-n 50))
(s/def ::line (s/nilable int?))
(s/def ::repo-id some?)
(s/def ::project-id some?)
(s/def ::external-review-id (specs/string-n 64))
(s/def ::suppression-details (s/keys :req-un [::suppressed-by ::reason ::line ::fn-name]
                                     :opt-un [::biomarker ::file-name ::external-review-id]))
(s/def ::suppression-flat (s/keys :req-un [::repo-id ::suppressed-by ::reason ::line ::fn-name ::biomarker ::file-name ::external-review-id]))
(s/def ::suppressions
  (s/map-of ::file-name (s/map-of ::biomarker (s/map-of ::fn-name ::suppression-details))))

(def key-fn (juxt :biomarker :file-name :fn-name :external-review-id))

(defn- db-obj [project-id repo-id details]
  (merge (map-of-db project-id repo-id) (->db-keys details)))

(s/fdef suppressions-tree
        :args (s/cat :suppressions (s/nilable (s/coll-of ::suppression-flat))
                     :external-review-id (s/nilable string?))
        :ret ::suppressions)
(defn suppressions-tree [flat-suppressions external-review-id]
  (let [suppressions (->> flat-suppressions
                          (filter #(contains? #{nil external-review-id} (:external-review-id %)))
                          (map #(dissoc % :project-id :repo-id :external-review-id))
                          distinct)]
    (group-by* [:file-name :biomarker :fn-name]
               {:fn->comp {:biomarker compare :file-name compare :fn-name compare}
                :acc-fn #(or %1 %2)}
               suppressions)))

(s/fdef suppressions-tree-for-statistics
  :args (s/cat :suppressions (s/nilable (s/coll-of ::suppression-flat)))
  :ret ::statistics-specs/suppressions)
(defn suppressions-tree-for-statistics [flat-suppressions]
  (->> flat-suppressions
       (map #(select-keys % [:repo-id :external-review-id :file-name :biomarker :fn-name]))
       distinct
       (group-by* [:repo-id :external-review-id :file-name :biomarker :fn-name]
                  {:acc-fn #(or %1 %2) :dissoc-keys? true})))

(defn load-suppressions [db-spec project-id repo-id]
  (->kebab (db-load-suppressions db-spec (map-of-db project-id repo-id))))

(defn load-suppressions-for-project [db-spec project-id]
  (->kebab (db-load-suppressions-by-project-id db-spec (map-of-db project-id))))

(defn ui->suppressions [pr-id reason user {:keys [name file method line]}]
  {:file-name file
   :biomarker name
   :reason reason
   :external-review-id pr-id
   :fn-name method
   :line line
   :suppressed-by user})

(defn add-suppress-warnings
  [db-spec {:keys [project-id repo-id user]} body]
  (let [{:keys [prId reason findings]} body
        db-suppressions (load-suppressions db-spec project-id repo-id)
        new-suppressions (mapv (partial ui->suppressions prId reason user) findings)]
    (log/info "Prev suppressions" (vec db-suppressions) " additional suppressions " new-suppressions)
    (let [{:keys [removed added updated]} (do-diff key-fn db-suppressions new-suppressions)]
      (doseq [_it removed] nil
        ;; if they are missing it's because we only get new ones
        #_(db-delete-suppression db-spec {:id (:id it)}))
      (doseq [it added]
        (let [updated-at (Date.)]
          (db-insert-suppression db-spec (db-obj project-id repo-id (assoc it :updated-at updated-at)))))
      (doseq [[old new] updated]
        (db-update-suppression db-spec (db-obj project-id repo-id (merge old new {:updated-at (Date.)})))))))
