(ns codescene.features.config.presets
  (:require [clojure.java.jdbc :as jdbc]
            [codescene.features.config.properties :as p]
            [codescene.features.util.coll :refer [do-diff]]
            [codescene.features.config.presets-db :as db]
            [codescene.features.delta.code-health.ch-config :as ch-config]
            [codescene.features.delta.coverage.cov-config :as cov-config]
            [medley.core :as m]))

(defmulti properties (fn [{:keys [category]}] category))
;; needs to be fixed
(defmulti find-uses (fn [{:keys [category]} _db-spec] category))

(defmethod find-uses :default
  [_ _] [])

(defn get-settings [db-spec id scope]
  (jdbc/with-db-transaction
    [tx db-spec]
    (when-let [main (db/get-settings tx id scope)]
      (assoc main
        :uses (find-uses main tx)
        :kv (p/from-db (properties main) (db/get-settings-kv tx id))))))

(def delete-settings db/delete-settings)

(defn params->bool [s]
  (case s
    nil nil
    ("true" true "1") true
    ("false" false "0") false))

(defn create-settings [{:keys [db-spec scope encryptor params html-form-input?]}]
  (jdbc/with-db-transaction
    [tx db-spec]
    (let [base {:category (:category params)
                :label (:label params)}
          id (db/insert-settings tx (merge base scope))]
      (run! (fn [[k v]]
              (db/insert-settings-kv tx id k v))
            (p/from-req-params (properties params) params encryptor html-form-input?))
      id)))

(def list-settings db/list-settings)

(defn available-settings [db-spec scope types]
  (->> (db/active-settings-by-categories db-spec types scope)
       (group-by :category)))

(defn settings-ids-by-category
  "Returns list of {:id, :active? } of settings by category"
  [db-spec category]
  (db/get-settings-ids-by-category db-spec category))

(defn update-settings [db-spec scope encryptor {:keys [id setting-id label active] :as params}]
  (jdbc/with-db-transaction
    [tx db-spec]
    (let [id (or id setting-id)
          base (m/assoc-some (db/get-settings tx id scope)
                             :label label
                             :active? (params->bool active))
          props (properties params)
          data (p/to-db props (p/from-req-params props params encryptor true))
          prev-data (update-keys (db/get-settings-kv tx id) name)
          {:keys [removed added updated] :as x} (do-diff first prev-data data)]
      (db/update-settings tx base)
      (run! (fn [[k _]] (db/delete-settings-kv tx id k))
            removed)
      (run! (fn [[k v]] (db/insert-settings-kv tx id k v))
            added)
      (run! (fn [[_old new]] (db/update-settings-kv tx id (first new) (second new)))
            updated)
      id)))

(defn invert-active [db-spec id]
  (jdbc/with-db-transaction
    [tx db-spec]
    (when-let [main (db/get-settings tx id {})]
      (db/update-settings tx (update main :active? not)))))

(defmethod properties "pr-quality-gates"
  [params]
  (mapv (fn [{:keys [id] :as p}]
          (if (= id :qg-preset) (assoc p :default "minimal") p))
        ch-config/qg-properties))

(defmethod properties "cov-config"
  [params]
  (mapv (fn [{:keys [id] :as p}]
          (if (= id :cov-qg-preset) (assoc p :default "custom") p))
        cov-config/settings-properties))

(defn new-properties [category]
  (p/from-db (properties {:category category}) {}))