(ns codescene.features.chat.service.llm.anthropic
  "Anthropic Claude API provider."
  (:require [clojure.data.json :as json]
            [clojure.string :as str]
            [codescene.features.chat.service.llm.http :as http]
            [codescene.features.chat.service.llm.provider :as provider]
            [taoensso.timbre :as log]))

(defn- extract-text-content
  "Extracts and concatenates text from Anthropic's content blocks."
  [response]
  (let [content-blocks (get-in response [:body :content])]
    (->> content-blocks
         (filter #(= (:type %) "text"))
         (map :text)
         (str/join ""))))

(defn- build-success-response
  "Builds a success response from Anthropic's API response format."
  [response]
  {:success true
   :content (extract-text-content response)
   :usage {:prompt-tokens (get-in response [:body :usage :input_tokens])
           :completion-tokens (get-in response [:body :usage :output_tokens])}})

(defn- build-error-response
  "Builds an error response from a failed API call."
  [response]
  {:success false
   :error (http/extract-error response (str "API error: " (:status response)))})

(defn- parse-response
  "Parses an Anthropic API response into our standard format."
  [response]
  (if (= 200 (:status response))
    (build-success-response response)
    (build-error-response response)))

(defn- extract-system-message
  "Extracts the system message content from a messages vector.
   Returns default if no system message is present."
  [messages]
  (let [system-msg (first (filter #(= :system (:role %)) messages))]
    (or (:content system-msg) "You are a helpful assistant.")))

(defn- remove-system-messages
  "Removes system messages from a messages vector.
   Anthropic requires system message as a separate parameter."
  [messages]
  (remove #(= :system (:role %)) messages))

(defrecord AnthropicProvider []
  provider/LLMProvider
  
  (call [_this {:keys [base-url chat-endpoint api-key model http-opts]} messages]
    (let [url (str base-url chat-endpoint)
          system-content (extract-system-message messages)
          other-msgs (remove-system-messages messages)]
      (log/debugf "Calling Anthropic API at %s" url)
      (let [response (http/post url
                                (merge {:headers {"x-api-key" api-key
                                                  "Content-Type" "application/json"
                                                  "anthropic-version" "2023-06-01"}
                                        :body (json/write-str {:model model
                                                               :max_tokens 4096
                                                               :system system-content
                                                               :messages (http/format-messages other-msgs)})
                                        :as :json}
                                       http-opts))]
        (if (http/request-failed? response)
          (http/format-request-error response)
          (parse-response response))))))
