(ns codescene.features.api.routes.safeguards
  "REST API routes for Code Health safeguards.
   Provides endpoints for Code Health quality gate insights and outcomes
   from various sources (PR analyses, MCP, IDE integrations).
   
   Route structure: /code-health/project/:id/safeguards/{source}/{endpoint}
   where source is one of: pr, mcp, ide
   and endpoint is one of: insights, outcomes"
  (:require [clojure.spec.alpha :as s]
            [codescene.features.api.safeguards :as safeguards]
            [codescene.features.api.privileges :as api-privileges]
            [codescene.features.api.core :as api-core]
            [codescene.features.api.spec.common :as common-spec]
            [codescene.features.components.ff :as ff]
            [codescene.features.util.maps :as maps]
            [compojure.api.sweet :refer [GET context]]
            [spec-tools.spec :as spec]))

(def ^:const outcomes-page-limit 100)
(s/def ::date_type (s/and string? #(not (empty? %)) #(re-matches common-spec/YYYY-MM-DD %)))

(defn pr-safeguards-routes
  "Routes for PR-based Code Health safeguard endpoints.
   Provides insights and outcomes from PR/delta analyses."
  [system project-id]
  (context "/pr" []
    (GET "/insights" req
      :middleware [#(api-core/wrap-authorize-project system % project-id #{api-privileges/restapi-read api-privileges/cli-access})]
      :summary "Get project Code Health safeguard insights from PR analyses"
      :tags ["code-health"]
      :query-params [{from :- ::date_type ""} {to :- ::date_type ""}]
      :responses {200 {:schema      spec/any?
                       :description "Code Health safeguard insights aggregated from PR analyses"}}
      (safeguards/insights system (maps/map-of project-id from to)))
    (GET "/outcomes" req
      :middleware [#(api-core/wrap-authorize-project system % project-id #{api-privileges/restapi-read api-privileges/cli-access})]
      :summary "Get project Code Health safeguard outcomes from PR analyses"
      :tags ["code-health"]
      :query-params [{from :- ::date_type ""} {to :- ::date_type ""}
                     {page :- spec/pos-int? 1}
                     {page-size :- spec/pos-int? outcomes-page-limit}]
      :responses {200 {:schema      spec/any?
                       :description "Individual PR outcomes for Code Health safeguards"}}
      (safeguards/outcomes system (maps/map-of project-id from to page page-size)))))

(defn safeguards-routes
  "Routes for all Code Health safeguard sources.
   Currently supports PR analyses, extensible to MCP and IDE sources."
  [system project-id]
  (context "/safeguards" []
    (pr-safeguards-routes system project-id)))

(defn sub-routes
  "Top-level routes for Code Health safeguards API.
   Feature flag controlled."
  [system]
  (context "/code-health" []
    (context "/projects/:project-id" [project-id]
      :path-params [project-id :- spec/pos-int?]
      (when (ff/on? (api-core/api-ff system) :code-health-safeguards-rest-api)
        (safeguards-routes system project-id)))))
