(ns codescene.features.api.routes.cli
  (:require
    [codescene.features.api.core :as api-core]
    [codescene.features.api.privileges :as api-privileges]
    [codescene.features.components.auth :as auth]
    [codescene.features.components.project :as project]
    [codescene.features.tools.licensing :as licensing]
    [codescene.features.util.api :as api-utils]
    [codescene.util.http :as http]
    [codescene.url.url-utils :as url-utils]
    [compojure.api.sweet :refer [context routes POST GET]]
    [taoensso.timbre :as log])
  (:import (clojure.lang ExceptionInfo)))

(defn- log-cli-authorize
  "Logs the fingerprint of the authorize request"
  [{:keys [body-params system] :as req}]
  (auth/mark-feature-use (api-core/api-auth system) req "cli-authorize" nil)
  (log/infof "authorize: %s" body-params)
  {:status 200})

(defn cli-license [{:keys [body-params system headers] :as req}]
  (log/infof "cli-license: %s" body-params)
  (auth/mark-feature-use (api-core/api-auth system) req "cli-license" (get headers "user-agent"))
  (let [user (auth/user (api-core/api-auth system) req)
        provider (api-core/api-licensing system)
        response (try
                   (http/response (licensing/cli-license provider user))
                   (catch ExceptionInfo e
                     (http/response (str "License unavailable: " (ex-message e)) 403)))]
    (http/plaintext-response response)))

(defn accesible-project-remote-repositories
  "Return a distinct(based on url and branch name) list of repositoriecodescene.url.url-utilss [{:url ... :branch ...} ...],
  branch can be nil, are skiped all local path url's"
  [{:keys [system] :as req}]
  (let [user (auth/user (api-core/api-auth system) req)
        projects-component (api-core/api-projects system)
        response (->> (project/all-accessible-projects projects-component user)
                      (map #(->> % :id (project/all-repositories projects-component)))
                      flatten
                      distinct
                      (filter #(-> % :url url-utils/is-repo-url?)))]
    (api-utils/ok response)))

(defn sub-routes
  [system]
  (routes
    (context "/cli" []
      :middleware [#(api-core/wrap-authorize-any system % #{api-privileges/restapi-read api-privileges/cli-access})]
      ;; TODO: Since Jun 2025 CLI uses licensing-routes, not this
      ;; we should remove when confident that there are no old CLI versions in use
      (POST "/authorize" request (log-cli-authorize request))
      (GET "/accesible-project-remote-repositories" request (accesible-project-remote-repositories request)))
    (context "/tool-license" []
      :middleware [#(api-core/wrap-authorize-any system % #{api-privileges/restapi-read api-privileges/cli-access})]
      (POST "/cli" request (cli-license request)))))