grammar Elixir;

translationunit
:
	expression* EOF
;

expression : multi_line_doc
           | multi_line_string
           | function
           | unit_test
           | module_references
           | anything;

assertion_entry // to parse test smells
:
	assert_expression* EOF
;

assert_expression : assertion_blocks
                  | anything;

assertion_blocks: assertion_statement+;

assertion_statement: ASSERT assertion_condition;
assertion_condition: ~(NEWLINE | '->')*? '->' function_body_token*? END NEWLINE* // assert_raise ..., fn -> ... end
                   | ~(NEWLINE)*? NEWLINE+; // all other assertions

anything: .;

module_references: alias | use;
alias: 'alias' alias_binding;
alias_binding: (aliased_module_part '.')+ aliased_module_part ',' 'as:' alias_name // alias Math.List, as: List
             | (aliased_module_part '.')+ alias_name NEWLINE // alias Math.List
             | (aliased_module_part '.')+ alias_tuple NEWLINE;  // alias TeslaMate.{Repo, Locations, Terrain, Settings}

alias_tuple: '{'  alias_tuple_destructuring_list+? '}';
alias_tuple_destructuring_list: alias_name
                              | alias_name ',' alias_tuple_destructuring_list;

aliased_module_part: SYMBOL;
alias_name: SYMBOL;

use: 'use' alias_name; // use Foo

multi_line_doc: MULTI_LINE_DOC_TAG ~MULTI_LINE_DOC_TAG*? MULTI_LINE_DOC_TAG;

function: (DEF | DEFP) function_name function_arguments? guard_clause body;
guard_clause: (logical_operator
               | ~(DO | DEF | DEFP))*?;
function_body_guard_clause: WHEN (logical_operator
                                  | ~(DO | DEF | DEFP | '->'))+;

unit_test: TEST unit_test_name test_parameters body;
unit_test_name: LITERAL | SYMBOL;
test_parameters: ~(DO | NEWLINE)*?;

body: single_line | multi_line;

single_line: ',' DO_COLON ~NEWLINE*? NEWLINE;

multi_line: DO function_body_token*? END;

function_body_token: string_sigils
                   | heex_template_sigil
                   | multi_line_string
                   | complexity
                   | anonymous_function_block
                   | block
                   | function_body_guard_clause
                   | logical_operator
                   | pattern_match_clause
                   | ~END;

anonymous_function_block: 'fn' function_body_token*? END;

complexity: if_block
          | single_line_if
          | case_construct
          | cond_construct
          | receive
          | try_catch
          | comprehension;

if_block: 'if' complexity_expr;
single_line_if: 'if' complexity_expr_condition*? single_line;

case_construct: 'case' complexity_expr;
pattern_match_clause: '->';
cond_construct: 'cond' complexity_expr_body;
receive: 'receive' complexity_expr_body;
try_catch: 'try' complexity_expr_body;

complexity_expr: complexity_expr_condition*? complexity_expr_body;
complexity_expr_condition: logical_operator
                         | ~(DO | DO_COLON);
logical_operator: AND | AND_OP | OR | OR_OP;
complexity_expr_body: DO NEWLINE* function_body_token+ END;

comprehension: FOR ~(DO_COLON | DO)+? DO_COLON
             | FOR ~(DO | DO_COLON | NEWLINE)+? complexity_expr_body; // for car <- list_cars() do \n ... end

block: DO ~END*? END;

string_sigils: STRING_SIGIL LeftParen ~RightParen*? RightParen;

heex_template_sigil: HEEx_TEAMPLATE_SIGIL heex_template;
heex_template: multi_line_string;
multi_line_string: MULTI_LINE_DOC_TAG .*? MULTI_LINE_DOC_TAG;

function_name: SYMBOL;

function_arguments: LeftParen argument_list? RightParen;
argument_list: argument
             | argument ','? argument_list;

argument: specific_argument_type arg_default_value?;
specific_argument_type: destructive_tuple_arg | simple_arg | list | map_argument | tuple_arg | binary_arg;
simple_arg: symbol_argument_type | ATOM | INT | float_type;
symbol_argument_type: '%'? SYMBOL;
float_type: INT DOT INT;

arg_default_value: default_value_marker specific_argument_type;
default_value_marker: DEFAULT_VALUE_MARKER | '=';

list: '[' list_argument_list*? ']';
list_argument_list: ~('[' | ']')
                  | list; // nested lists (destructuring)

map_argument: '%' '{' map_key arrow? simple_arg '}' map_argument_bind_name?;
arrow: '=>';
map_key: keyword_arg | LITERAL | SINGLE_LITERAL;
keyword_arg: (simple_arg ':') | (':' simple_arg);
map_argument_bind_name: '=' simple_arg;

destructive_tuple_arg: symbol_argument_type tuple_arg;
tuple_arg: '{'  tuple_argument_list*? '}';
tuple_argument_list: ~('{' | '}')
                   | tuple_arg; // nested tuples (destructuring)

binary_arg: '<<' binary_argument_list*? '>>';
binary_argument_list: ~('<<' | '>>')
                    | binary_arg;

HEEx_TEAMPLATE_SIGIL: '~H';

MODULE_DOC: '@moduledoc';
MULTI_LINE_DOC_TAG: '"""';

fragment ESCAPED : '\\\\' | '\\"';
LITERAL : '"' ( ESCAPED | ~('\n'|'\r') )*? '"';

SINGLE_LITERAL : '\''  ~('\n'|'\r')*? '\'';

STRING_SIGIL: '~'[sScw];

fragment ESCAPED_SLASH : '\\/';
REGEX : '~r/' (ESCAPED_SLASH | ~('\n'|'\r') )*? '/';

LineComment: '#' ~'{' ~[\r\n]* -> skip;

LeftParen : '(';
RightParen: ')';

WHEN: 'when';
DEFP: 'defp'; // private function
DEF: 'def';
DO_COLON: 'do:';
DO:'do';
END: 'end';
FOR: 'for';
AND: 'and';
AND_OP: '&&';
OR: 'or';
OR_OP: '||';
VERTICAL_BAR: '|'; // e.g. for list destructuring
DEFAULT_VALUE_MARKER: '\\\\';

// Ex Unit
TEST: 'test';
ASSERT: 'assert' ('_' [a-z_]+)?;

INT: [0-9]+;
DOT: '.';
ATOM: ':' [*+!\-_?>a-zA-Z][*+!\-_?a-zA-Z0-9]*;
SYMBOL: [*+!\-_?>a-zA-Z][*+!\-_?a-zA-Z0-9]*;

Whitespace : [ \t]+ -> skip;
NEWLINE : '\r'? '\n';

ANY_CHAR : .; // Put this lexer rule last to give it the lowest precedence
