(ns codescene.features.scheduling.ephemeral
  (:require [overtone.at-at :as at-at]
            [taoensso.timbre :as log]))

(def ^:dynamic *thread-pool* nil)

(defn init
  "Initialize thread pool and prepare for scheduling that is not persistent,
  and will run on each node in a cluster. This is suitable for tasks like
  cleanup of local folders."
  []
  (log/info "Initializing thread pool and preparing ephemeral per-node scheduling")
  (alter-var-root #'*thread-pool* (constantly (at-at/mk-pool))))

(defn destroy
  "Stop all threads and shutdown the thread pool, preparing for application shutdown or
  a new call to init."
  []
  (log/info "Stopping scheduled ephemeral per-node jobs and destroying thread pool")
  (if *thread-pool*
    (let [jobs (at-at/scheduled-jobs *thread-pool*)
          n (count jobs)]
      (when (pos? n)
        (log/infof "Stopping %d scheduled jobs gracefully" n)
        (doseq [{:keys [id desc]} jobs]
          (log/debugf "Stopping job %s with :desc \"%s\"" id desc)
          (at-at/stop id *thread-pool*))
        (Thread/sleep 1000)

        ;; give the jobs a few chances to stop gracefully
        (loop [n-still-running (count (at-at/scheduled-jobs *thread-pool*))
               tries 3]
          (when (and (pos? tries)
                     (pos? n-still-running))
            (log/infof "%d jobs still running, waiting for jobs to stop" n-still-running)
            (Thread/sleep 3000)
            (recur (count (at-at/scheduled-jobs *thread-pool*)) (dec tries))))

        ;; kill any remaining jobs
        (when (pos? (count (at-at/scheduled-jobs *thread-pool*)))
          (log/infof "Killing %d scheduled jobs" n)
          (doseq [{:keys [id desc]} jobs]
            (log/debugf "Killing job %s with :desc \"%s\"" id desc)
            (at-at/kill id *thread-pool*))))

      (log/info "Shutting down thread pool" *thread-pool*)
      (.shutdownNow (:thread-pool @(:pool-atom *thread-pool*)))
      (alter-var-root #'*thread-pool* (constantly nil)))
    (log/debug "No thread pool to destroy")))

(defn recurring-job
  "Calls `f` repeatedly with an interspacing of `interval-ms`
  i.e. the next call of `f` will happen `interval-ms` milliseconds
  after the completion of the previous call."
  [f interval-ms description]
  (at-at/interspaced interval-ms
                     f
                     *thread-pool*
                     :desc description))
