(ns codescene.features.repository-provider.gerrit.delta
  (:require [clojure.string :as str]
            [codescene.features.delta.result.pull-request :as result]
            [codescene.features.delta.file-regions :as file-regions]
            [codescene.features.delta.protocols :as protocols]
            [codescene.features.delta.pr-status :as pr-status]
            [codescene.features.project.pr-integration :as pi]
            [codescene.features.repository-provider.gerrit.api :as api]
            [codescene.features.repository-provider.providers :as providers]
            [codescene.features.util.template :as template]
            [codescene.url.url-utils :as url]
            [medley.core :as m]
            [taoensso.timbre :as log]))

(defn- strip-html
  "Naive HTML stripping, sufficient to remove html tags we put in our comments, because
  Gerrit markdown doesn't support HTML tags."
  [s] (str/replace s #"<[a-zA-Z/]+>" ""))

(defn finding->comment [{:keys [description title-prefix category file-name category-description location] :as ctx}
                        markdown?]
  (let [{:keys [start end after?]} location
        title (str title-prefix " " category)]
    (m/assoc-some
      {:robot_id "CodeScene"
       :robot_run_id (str "codescene-" (random-uuid))
       :path file-name
       :message (if markdown?
                  (strip-html (template/render-file "templates/delta/file-comment.md" ctx))
                  (cond-> (str title "\n" description)
                    category-description (str ". " category-description)))
       :side (if after? "REVISION" "PARENT")}
      :range (when start {:start_line start
                          :start_character 0
                          :end_character 0
                          :end_line end}))))

(defn supports-markdown? [authed-client]
  (let [[major minor] (api/get-version authed-client)]
    (and major minor (or (and (= major 3) (>= minor 7))
                         (> major 3)))))

(defn review-comments [pr-presentable markdown? locator]
  (->> (result/file-comment-items pr-presentable locator)
       (map #(finding->comment % markdown?))
       (group-by :path)))

(defn comment-poster
  [delta-authed-client-fn {:keys [full-changes-id revision-id] :as provider-ref}]
  (let [authed-client (delta-authed-client-fn provider-ref)]
    {:markdown? (supports-markdown? authed-client)
     :post-review #(api/set-review authed-client full-changes-id revision-id %1 %2)}))

(defn post-delta-results
  "Posts the results of the delta: the main comment, then any file comments if enabled."
  [delta-authed-client-fn {:keys [result-options] :as provider-ref} pr-presentable]
  (log/debugf "provider=gerrit; action=delta-analysis-results; ref=%s" provider-ref)
  (let [locator (result/cached-locator #(comp first (file-regions/change-locator % false)))
        {:keys [comments?]} result-options
        {:keys [markdown? post-review]} (comment-poster delta-authed-client-fn provider-ref)
        comment-fn (if markdown?
                     #(result/presentable->comment %1 :gerrit %2)
                     result/presentable->text-comment)]
    (when (result/comment? pr-presentable result-options)
      (post-review (comment-fn pr-presentable (not comments?))
                   (when comments? (review-comments pr-presentable markdown? locator))))))

(defn post-delta-skipped [delta-authed-client-fn provider-ref reason result-url]
  (log/debugf "provider=gerrit; action=delta-analysis-skipped; ref=%s" provider-ref)
  (let [{:keys [markdown? post-review]} (comment-poster delta-authed-client-fn provider-ref)
        fmt-str (if markdown?
                  "### CodeScene PR Check Skipped\n%s\n---\n%s"
                  "CodeScene PR Check Skipped\n\n%s\n\n%s")]
    (when (or (-> provider-ref :result-options :always-comment?) (providers/negative-skip? reason))
      (post-review (format fmt-str
                           (result/render-error-title reason)
                           (result/render-error reason))
                   nil))))

(defn post-delta-error [delta-authed-client-fn provider-ref error result-url]
  (log/debugf "provider=gerrit; action=delta-analysis-error; ref=%s" provider-ref)
  (let [{:keys [markdown? post-review]} (comment-poster delta-authed-client-fn provider-ref)
        fmt-str (if markdown?
                  "### CodeScene PR Check\n%s\n---\n%s"
                  "CodeScene PR Check\n\n%s\n\n%s")]
    (post-review (format fmt-str
                         (result/render-error-title error)
                         (result/render-error error))
                 nil)))

(defn post-delta-pending
  "Does nothing so far."
  [delta-authed-client-fn provider-ref result-url]
  (log/debugf "provider=gerrit; action=delta-analysis-pending; ref=%s" provider-ref)
  nil)

(defrecord GerritDelta [delta-authed-client-fn]
  protocols/DeltaResultBoundary
  (delta-pending [this provider-ref result-url]
    (post-delta-pending delta-authed-client-fn provider-ref result-url))
  (delta-results [this provider-ref pr-presentable]
    (post-delta-results delta-authed-client-fn provider-ref pr-presentable))
  (delta-skipped [this provider-ref reason result-url]
    (post-delta-skipped delta-authed-client-fn provider-ref reason result-url))
  (delta-error [this provider-ref error result-url]
    (post-delta-error delta-authed-client-fn provider-ref error result-url)))

(defn repo-id [change] (str (:host (url/repo-url->parts (:url change))) "//" (:project change)))

(defn process-merged-event [tx project-id body]
  (log/infof "Change merged, marking it closed repo-id=%s change-id=%s project-id=%s"
             (repo-id (:change body))
             (get-in body [:change :number])
             project-id)
  (pr-status/delete-open-pr
    tx
    project-id
    (repo-id (:change body))
    (str (get-in body [:change :number]))))

(defn- valid-hook? [{:keys [enabled? events]}]
  (and enabled? (#{"patchset-created" "change-merged"} (set events))))

(defn webhook-status
  [hooks]
  (cond
    (and (= 1 (count hooks))
         (every? valid-hook? hooks))
    :installed
    (zero? (count hooks))
    :not-installed
    :else
    :invalid))

(defn repository-webhook-status
  [pr-integration repositories]
  (let [our-hook? (pi/our-hook-fn? pr-integration)
        authed-client (pi/-authed-client pr-integration (first repositories))
        ;; TODO Rok use authed client for each repo separately
        hooks (api/get-repos-webhooks authed-client repositories our-hook?)]
    (mapv #(assoc {} :hooks %
                     :status (webhook-status %)) hooks)))

(defrecord GerritHookAccess []
  protocols/HookAccess
  (-repos-hooks [this PrIntegrationConfig repositories]
    (repository-webhook-status PrIntegrationConfig repositories))
  (-add-hooks [this PrIntegrationConfig repositories]
    (let [repo-path (fn [{:keys [url]}]
                      (-> url url/repo-url->parts :repo-path api/normalize-repo-path))]
      (mapv (fn [repo]
              (log/infof "Installing Gerrit webhook to %s" (:url repo))
              [(api/create-repo-webhook (pi/-authed-client PrIntegrationConfig repo)
                                        (repo-path repo)
                                        (pi/-callback-url PrIntegrationConfig)
                                        ["patchset-created" "change-merged"])])
            repositories)))
  (-remove-hooks [this PrIntegrationConfig repos]
    (doseq [repo repos
            :let [authed-client (pi/-authed-client PrIntegrationConfig repo)]
            hook (:hooks repo)]
      (log/infof "Removing Gerrit webhook from %s" (:url hook))
      (api/delete-entity authed-client (:url hook)))))