(ns codescene.features.reports.graphics.functions
  (:require
    [codescene.features.reports.graphics.doughnut :as doughnut]
    [codescene.features.reports.graphics.chart :as chart]
    [codescene.features.reports.common.styling :as styling]
    [codescene.features.reports.common.style-def :as style-def]
    [codescene.util.time :as ut]
    [tick.alpha.interval :as ti]
    [tick.core :as t])
  (:import [java.awt BasicStroke]
           (org.jfree.chart JFreeChart)))


(defn- after? [{:keys [date]} that]
  (let [this (ut/instant date)]
    (ti/preceded-by? this that)))

(defn- pick-dot-data [trend]
  (let [reversed-trend (reverse trend)]
    (loop [remaining-trend (rest reversed-trend)
           latest-item (first reversed-trend)
           ret []]
      (if latest-item
        (let [latest-instant (ut/instant (:date latest-item))
              day-before-yesterday (t/<< latest-instant (t/new-period 2 :days))
              new-trend (drop-while #(after? % day-before-yesterday) remaining-trend)]
          (recur (rest new-trend) (first new-trend) (conj ret latest-item)))
        (reverse ret)))))

(defn trend->dot-data [trend]
  (let [dot-data (pick-dot-data trend)]
    (if (or (= 1 (count dot-data))
            (= (first dot-data) (first trend)))
      dot-data
      (cons (first trend) (rest dot-data)))))


(defn- split-into-categories [dot-data [lower upper]]
  (let [risky-data (filter #(< (:value %) lower) dot-data)
        attention-data (filter #(< lower (:value %) upper) dot-data)
        healthy-data (filter #(< upper (:value %)) dot-data)]
    {:risky-data     risky-data
     :attention-data attention-data
     :healthy-data   healthy-data}))


(defn code-health-chart-fn
  "
  trend: a sequence of {:value <double> :date <string>} ordered by :date ascending.
  score-bounds: a sequence of two doubles, lower and upper bound, separating risky/attention/healthy categories
  Calculates the data necessary for the code health trend chart and returns a function for drawing the chart.
  "
  [trend score-bounds beginning end]
  (fn [g2d width height]
    (let [dot-data (trend->dot-data trend)
          category-data (split-into-categories dot-data score-bounds)
          all-data (assoc category-data :line-data trend)
          ^JFreeChart chart (chart/code-health-trend all-data score-bounds beginning end)]
      (.draw chart g2d (java.awt.Rectangle. -10 0 (+ 20 width) height)))))


(defn code-health-overview-doughnut-fn
  [{:keys [red-value yellow-value green-value]}]
  (fn [g2d width height]
    (let [value-1 (if (number? green-value) green-value 0)
          value-2 (if (number? yellow-value) yellow-value 0)
          value-3 (if (number? red-value) red-value 0)
          some-values? (pos? (+ value-1 value-2 value-3))]
      (doughnut/draw-doughnut
        g2d width height
        {:doughnut-options
         {:values       [(if some-values? value-1 10) value-2 value-3]
          :colors       [(if some-values? style-def/healthy-color style-def/light-grey) style-def/attention-color style-def/risky-color]
          :end-cap      BasicStroke/CAP_BUTT
          :stroke-width 8}}))))


(defn code-health-doughnut-fn
  [{:keys [code-health fill-color]}]
  (fn [g2d width height]
    (let [value-1 (if (number? code-health) code-health 0)]
      (doughnut/draw-doughnut
        g2d width height
        {:doughnut-options
         {:values       [value-1 (- 10 value-1)]
          :colors       [fill-color style-def/light-grey]
          :end-cap      BasicStroke/CAP_SQUARE
          :stroke-width 2}
         :text-options
         {:text       (styling/number->cutoff-string code-health)
          :text-color style-def/dark-grey
          :font-style java.awt.Font/PLAIN
          :font-size  12
          :nudge-text 4}}))))

(defn pr-statistics-doughnut-fn
  [{:keys [red-value green-value grey-value text]}]
  (let [value-1 (if (number? green-value) green-value 0)
        value-2 (if (number? red-value) red-value 0)
        value-3 (if (number? grey-value) grey-value 0)
        some-values? (pos? (+ value-1 value-2 value-3))]
    (fn [g2d width height]
      (doughnut/draw-doughnut
        g2d width height
        {:doughnut-options
         {:values       [(if some-values? value-1 10) value-2 value-3]
          :colors       [(if some-values? style-def/healthy-color style-def/light-grey) style-def/risky-color style-def/font-lightgrey]
          :end-cap      BasicStroke/CAP_BUTT
          :stroke-width 12}
         :text-options
         {:text       text
          :text-color style-def/dark-grey
          :font-style java.awt.Font/PLAIN
          :font-size  8
          :nudge-text 0}}))))
