(ns codescene.features.pm-data.clickup.ticket-id-extractor
  (:require [clojure.string :as str]
            [codescene.features.pm-data.extractor-common :as ec]
            [codescene.pm-data.provider-common :as c]
            [medley.core :as m]))

;; match internal ids referenced like CU-{ticket-id} (eg CU-86bz8n71p)
;; as well as custom ids reference like {prefix}-{nbr} (eg CS-29)
;; with internal id in group 1, custom id in group 2
(def ticket-id-pattern #"(?:CU-([\w-]+)|([\w-]+-\d+))") 
 
(defn match->ticket-id
  [[_match internal-task-id custom-task-id]]
  (if internal-task-id
    {:number internal-task-id}
    (let [[scope number] (str/split custom-task-id #"-")]
      {:scope scope
       :number number})))

(defn extract-ticket-ids
  [msg]
  (->> msg
       (re-seq ticket-id-pattern)
       (map match->ticket-id)))
 
(defn scoped-ticket-id [{:keys [scope number] :as _ticket-id}]
  (if scope
    (format "%s-%s" (str/upper-case scope) number)
    number))

(defn with-resolved-aliases
  "Resolve a ticket-id match using the proved aliases.
   Note that keys are looked up in upper case"
  [aliases {:keys [scope] :as ticket-id}]
  (if scope
    (ec/with-resolved-aliases aliases ticket-id)
    ticket-id))

(defn- make-parent-lookup [provider-def internal-id->id issues]
  (if (:map-subtasks-to-parent-issues provider-def)
    (->> issues
         (keep (fn [{:keys [id parent]}]
                 (when parent
                   [id (internal-id->id parent)])))
         (into {}))
    {}))

(defn- resolve-internal-ids [internal-id->id ticket-ids]
  (map #(get internal-id->id % %) ticket-ids))

(defn- resolve-parents [->parent ticket-ids]
  (map #(get ->parent % %) ticket-ids))

(defn make-ticket-id-extractor
  [{:keys [external-project-ids] :as provider-def} issues]
  (let [aliases (ec/aliases-from provider-def)
        ->pr-ticket-ids (ec/make-pr-ticket-ids-lookup provider-def extract-ticket-ids)
        internal-id->id (->> (m/index-by :internal-id issues)
                             (m/map-vals :id))
        ->parent (make-parent-lookup provider-def internal-id->id issues)]
    (fn [rev _date-time msg]
      (when (seq external-project-ids)
        (->> (concat (extract-ticket-ids msg)
                     (->pr-ticket-ids (subs rev 0 7)))
             ;;(filter (ec/make-ticket-filter external-project-ids aliases))
             (map #(with-resolved-aliases aliases %))
             (map scoped-ticket-id)
             (resolve-internal-ids internal-id->id)
             (resolve-parents ->parent))))))

(comment
  (def api-client (codescene.features.pm-data.clickup.api/clickup-auth (System/getenv "CLICKUP_TOKEN")))
  (def provider-def {:api-client  api-client
                     :external-project-ids ["901505265065"]
                     :project-aliases [["C" "CS"]]
                     :map-subtasks-to-parent-issues true})
  (def extractor (make-ticket-id-extractor provider-def []))
  (extractor "1234567890" "date not used" "a CS-123 b")
  )
