(ns codescene.features.chat.service.prompt.parser
  "LLM response parsing for chat service.
   
   Parses structured JSON responses from LLMs into Clojure data."
  (:require [clojure.data.json :as json]
            [clojure.string :as str]
            [taoensso.timbre :as log]))

(defn- strip-code-fences
  "Removes markdown code fence markers from JSON content."
  [content]
  (-> content
      (str/replace #"^```json?\s*" "")
      (str/replace #"\s*```$" "")
      str/trim))

(defn- parse-json
  "Attempts to parse content as JSON.
   Returns parsed map or nil on failure."
  [content]
  (try
    (json/read-str content :key-fn keyword)
    (catch Exception _
      nil)))

(defn- build-parsed-response
  "Builds response map from parsed JSON data."
  [parsed]
  {:success true
   :thinking (:thinking parsed)
   :tool-call (:tool_call parsed)
   :response (:response parsed)})

(defn- build-plain-text-response
  "Builds response map for plain text (non-JSON) content."
  [content]
  {:success true
   :tool-call nil
   :response content})

(defn parse-llm-response
  "Parses the structured JSON response from the LLM."
  [content]
  (let [cleaned (strip-code-fences content)]
    (if-let [parsed (parse-json cleaned)]
      (build-parsed-response parsed)
      (do
        (log/debug "Failed to parse as JSON, treating as plain text")
        (build-plain-text-response content)))))
