(ns codescene.features.middleware
  (:require
   [codescene.cache.core :as cache]
   [codescene.util.http :as http]
   [medley.core :as m]
   [taoensso.timbre :as log])
  (:import
   (org.apache.commons.lang3 RandomStringUtils)))

(defn wrap-scoped-cache [handler]
  (fn
    ([request]
     (cache/with-scoped-cache
       (handler request)))
    ([request respond raise]
     (cache/with-scoped-cache
       (handler request respond raise)))))

(defn- generate-request-id []
  (.nextAlphabetic (RandomStringUtils/insecure) 8))

(defn- enriched-request-context [request]
  (m/assoc-some log/*context*
                :request-id (generate-request-id)
                :client-ip (http/get-client-ip request)
                ;; See https://app.clickup.com/t/9015696197/CS-4838 for more info about these fields
                :cloudfront-request-id (get-in request [:headers "x-amz-cf-id"])
                :alb-request-id (get-in request [:headers "x-amzn-trace-id"])
                :client-country (get-in request [:headers "cloudfront-viewer-country"])
                :client-region (get-in request [:headers "cloudfront-viewer-country-region"])))

(defn wrap-request-logging
  "Creates and adds a request ID and adds it to logging context
  and also client-ip as determined by the x-forwarded-for header."
  [handler]
  (fn
    ([request]
     (log/with-context (enriched-request-context request)
                       (handler request)))
    ([request respond raise]
     (log/with-context (enriched-request-context request)
                       (handler request respond raise)))))
