(ns codescene.features.factors.kpi-time-series
  (:require
    [clj-time.core :as t]
    [clj-time.format :as f]
    [clojure.java.io :as io]
    [clojure.java.jdbc :as jdbc]
    [codescene.analysis.paths :as paths]
    [codescene.factors.code-health.presentation :as code-health-presentation]
    [codescene.factors.delivery.presentation :as delivery-presentation]
    [codescene.factors.core :refer [kpi-type->path kpi-types]]
    [codescene.features.dashboard.core :as dashboard-common]
    [codescene.features.util.maps :as maps]
    [codescene.features.util.string :refer [declob]]
    [codescene.util.csv-conversions :as csv-conversions]
    [codescene.util.json :as json]
    [codescene.util.numeric :refer [safe-subtract]]
    [hugsql.core :as hugsql]
    [medley.core :as m]
    [taoensso.encore :as encore]
    [taoensso.tufte :as tufte :refer [defnp]])
  (:import
    java.io.FileNotFoundException))

(hugsql/def-db-fns "codescene/features/factors/kpi-time-series.sql")
(declare time-series-by-project-id*)
(declare insert-time-series*)
(declare delete-time-series*)
(declare time-series-by-analysis-id*)

(defn analysis->time-series-record [project-id analysis-id results-absolute-path kpi-type]
  (try
    {:kpi-time-series (seq (json/read (str results-absolute-path "/" (kpi-type kpi-type->path))))
     :kpi-type (name kpi-type)
     :project-id project-id
     :analysis-id analysis-id}
    (catch FileNotFoundException _ nil)))

;; (s/def ::kpi-time-series-data (s/coll-of (s/keys :req-un [::date ::type])))
;; (s/def ::kpi-time-series-record (s/keys :req-un [::project-id ::analysis-id ::kpi-type ::kpi-time-series]))

(defn ->insertable [kpi-time-series-record]
  (-> kpi-time-series-record
      (update :kpi-time-series json/generate-string)))

(defn ->readable [kpi-time-series-from-db]
  (-> kpi-time-series-from-db
      maps/->kebab
      (update :kpi-time-series declob)
      (update :kpi-time-series json/parse-string)
      (update :kpi-type keyword)))

(defn replace-time-series
  [db-spec time-series-record]
  (jdbc/with-db-transaction [tx db-spec]
                            (delete-time-series* tx time-series-record)
                            (insert-time-series* tx time-series-record)))

;; (s/fdef save-time-series
;;   :args (s/cat :this ::kpi-time-series-db
;;                :absolute-analysis-path ::cs-specs/non-empty-string))
(defn update-time-series [db-spec project-id analysis-id absolute-analysis-path]
  (doall (->> kpi-types
              (map (partial analysis->time-series-record project-id analysis-id absolute-analysis-path))
              (map ->insertable)
              (map (partial replace-time-series db-spec)))))

(defn time-series-by-project-id [db-spec project-id]
  (map ->readable (time-series-by-project-id* db-spec {:project-id project-id})))

(defn time-series-by-analysis-id
  [db-spec project-id analysis-id]
  (into {}
        (comp
         (map ->readable)
         (map (juxt :kpi-type :kpi-time-series)))
        (time-series-by-analysis-id* db-spec {:project_id project-id :analysis_id analysis-id})))

(defn- by-kpi-type [all-kpis kpi-type]
  (hash-map kpi-type
            (some (fn [kpi] (when (= kpi-type (:kpi-type kpi)) kpi)) all-kpis)))

(defn all-kpis-for-project!
  ;; TODO @aasa spec data format
  "Returns a single map of kpi-type the corresponding time series data:
  {:code-health [{:date ... :kpi ... } ... n]
   :delivery ... n}."
  [db-spec project-id]
  (let [all-kpis (time-series-by-project-id db-spec project-id)
        kpi-types (map :kpi-type all-kpis)]
    (into {}
          (map
           (partial by-kpi-type all-kpis)
           kpi-types))))

(defn- remove-delivery-details
  [delivery-kpi]
  (-> delivery-kpi
      (update-in [:kpis :day] select-keys  [:kpi :sub-kpis])
      (update-in [:kpis :week] select-keys [:kpi :sub-kpis])
      (update-in [:kpis :month] select-keys [:kpi :sub-kpis])
      (update-in [:kpis :year] select-keys  [:kpi :sub-kpis])))

(defn- remove-sub-kpi-name
  [kpi-time-series]
  (update kpi-time-series :sub-kpis (fn [sub-kpis]
                                      (map #(dissoc % :name) sub-kpis))))

(defn- remove-details
  [kpi-time-series]
  (map
   (fn pruning [{:keys [kpis] :as i}]
     (cond-> (select-keys i [:date :kpi :sub-kpis :kpis :error :delta])
       kpis remove-delivery-details
       true remove-sub-kpi-name))
   kpi-time-series))

(defn remove-error-data-points
  "Remove all data points that are not valid.

  To display misconfiguration we instead use the config diagnostic object, rather than
  by reading error data points."
  [kpi-time-series]
  (remove #(contains? % :error) kpi-time-series))

(def iso-date-formatter (f/formatter "yyyy-MM-dd"))

(def date->string (partial f/unparse iso-date-formatter))

(defn- time-series-start-date
  "The idea is to choose a start date that gives us about a month worth of data.
  But we also include an extra datapoint before this cut-off date, because
  we don't want the graph to appear cut off."
  [time-series-dates]
  (encore/when-let [last-date (some-> time-series-dates last f/parse)
                    approx-start-date (date->string (t/minus last-date (t/months 1)))
                    [before after] (split-with #(neg? (compare % approx-start-date)) time-series-dates)]
     (cond
       (> (count before) 0) (last before)
       :else (first after))))

(defn- ->date-series
  [kpi-time-series]
  (sort (map :date kpi-time-series)))

(defn select-data-points-in-scope
  "Selects a slice of the time series by deriving a start and end date
  from the last element.

  Other values are removed, taking into account not everything in the time
  series may be in order due to historical mishaps.

  Comparisons of dates are done alphabetically."
  [kpi-time-series]
  (encore/when-let [sorted-dates (->date-series kpi-time-series)
                    start-date (time-series-start-date sorted-dates)
                    end-date (last sorted-dates)]
    (->> kpi-time-series
         (remove #(pos? (compare (:date %) end-date)))
         (remove #(neg? (compare (:date %) start-date))))))

(defn prune-project-keys-for-projects-page
  [project]
  ;; TODO: this might not be the best strategy for narrowing
  ;; the data here. For now, this helps avoid some
  ;; unnecessary CLOBs in on-prem that sneak into the project config.
  (select-keys project [:name
                        :description
                        :id
                        :last-analysis
                        :analysis-data
                        :analysis-destination
                        :groups
                        :group-id
                        :run-status
                        :pr-integration
                        :demo-project?]))

(defn- total-lines-of-code
  [analysis-directory]
  (let [csv-file (io/file analysis-directory paths/dashboard-csv)]
    (dashboard-common/application-code-loc csv-file)))

(defn- active-authors [analysis-directory]
  (let [csv-file (io/file analysis-directory paths/dashboard-csv)
        dashboard-data (csv-conversions/optional-csv->map csv-file)]
    (:activeauthors dashboard-data)))

(defnp kpis-for-projects-page
  "Analysis job has keys :id, :analysis-time, :dir."
  [project analysis-job-with-dir kpis]
  (-> project
      (assoc :last-analysis (assoc (select-keys analysis-job-with-dir [:id :analysis-time])
                              :total-lines-of-code (total-lines-of-code (:dir analysis-job-with-dir))
                              :active-authors (active-authors (:dir analysis-job-with-dir))))
      (assoc :analysis-data kpis)))

(defn extract-sub-kpi-value
  "Extract the value for a specific sub-kpi key from a sample's sub-kpis collection.
   Sub-kpis have :key as string (e.g. 'average-code-health') and :value as the numeric value."
  [sub-kpis sub-kpi-key]
  (when-let [found (m/find-first #(= (name sub-kpi-key) (:key %)) sub-kpis)]
    (:value found)))

(defn find-sample-closest-to-date
  "Find the sample closest to (but not after) the target date.
   Samples are expected to have a :date key in 'yyyy-MM-dd' format.
   Returns nil if no valid sample is found."
  [time-series target-date-str]
  (when (seq time-series)
    (->> time-series
         (filter :date)
         (filter #(not (pos? (compare (:date %) target-date-str))))
         (sort-by :date)
         last)))

(defn- date-minus-period
  "Given a DateTime, subtract a period and return formatted date string."
  [date-time period]
  (f/unparse iso-date-formatter (t/minus date-time period)))

(defn extract-code-health-at-timepoints
  "Extract code health values (now, month-back, year-back) from time series for a given sub-kpi.
   Returns a map with :now, :month, :year keys containing the values (or nil if not found),
   plus :change_month and :change_year containing the delta from those timepoints to now.
   
   Arguments:
   - time-series: sequence of samples with :date and :sub-kpis keys
   - sub-kpi-key: keyword like :average-code-health or :hotspots-code-health
   - now-date: clj-time DateTime representing the analysis end time"
  [time-series sub-kpi-key now-date]
  (let [now-str (f/unparse iso-date-formatter now-date)
        month-back-str (date-minus-period now-date (t/months 1))
        year-back-str (date-minus-period now-date (t/years 1))
        now-sample (find-sample-closest-to-date time-series now-str)
        month-sample (find-sample-closest-to-date time-series month-back-str)
        year-sample (find-sample-closest-to-date time-series year-back-str)
        now-value (extract-sub-kpi-value (:sub-kpis now-sample) sub-kpi-key)
        month-value (extract-sub-kpi-value (:sub-kpis month-sample) sub-kpi-key)
        year-value (extract-sub-kpi-value (:sub-kpis year-sample) sub-kpi-key)]
    {:now now-value
     :month month-value
     :year year-value
     :change_month (safe-subtract now-value month-value)
     :change_year (safe-subtract now-value year-value)}))

(defn build-code-health-summary
  "Build code health summary from KPI time series data.
   Returns a map with :code-health and :hotspot-code-health, each containing
   :now, :month, :year values.
   
   - :code-health uses average-code-health (full codebase weighted average)
   - :hotspot-code-health uses hotspots-code-health (hotspots only)
   
   Arguments:
   - code-health-series: the :code-health time series from the DB
   - now-date: clj-time DateTime representing the analysis end time"
  [code-health-series now-date]
  {:code-health (extract-code-health-at-timepoints code-health-series
                                                   code-health-presentation/average-code-health-key
                                                   now-date)
   :hotspot-code-health (extract-code-health-at-timepoints code-health-series
                                                           code-health-presentation/hotspots-code-health-key
                                                           now-date)})

(defn- extract-delivery-sub-kpi
  "Extract a specific sub-kpi value from delivery sample's monthly scope.
   Delivery samples have a nested structure: {:kpis {:month {:sub-kpis [...]}}}
   The :key in sub-kpis can be either a keyword or string depending on source."
  [sample sub-kpi-key]
  (let [sub-kpis (get-in sample [:kpis :month :sub-kpis])
        key-name (name sub-kpi-key)]
    (when-let [found (m/find-first #(= key-name (name (:key %))) sub-kpis)]
      (get-in found [:presentation :value]))))

(defn build-delivery-summary
  "Build delivery summary from KPI time series data.
   Returns a map with :unplanned-work-percent and :avg-development-time-minutes
   extracted from the monthly scope of the latest delivery sample.
   
   Returns nil if delivery data is not available (requires PM integration
   with cycle-time estimation enabled).
   
   Arguments:
   - delivery-series: the :delivery-performance time series from the DB"
  [delivery-series]
  (when-let [latest-sample (last delivery-series)]
    (let [unplanned-work (extract-delivery-sub-kpi latest-sample delivery-presentation/unplanned-work-key)
          development-time (extract-delivery-sub-kpi latest-sample delivery-presentation/development-time-key)]
      (when (or unplanned-work development-time)
        {:unplanned-work-percent unplanned-work
         :avg-development-time-minutes development-time}))))
