(ns codescene.features.chat.service
  "Generic chat service implementation.
   
   This is a BYOK (Bring Your Own Key) implementation that:
   - Accepts LLM configuration from the user (provider, model, API key)
   - Proxies requests to the user's chosen LLM provider
   - Integrates with CodeScene's MCP server for tool execution
   
   Architecture:
   - LLM providers are simple prompt-in/response-out
   - Tools are described in the system prompt
   - LLM responds with structured JSON including any tool calls
   - We execute tools and make a follow-up call with results"
  (:require [codescene.features.chat.service.loop :as loop]
            [codescene.features.chat.service.mcp :as mcp]
            [codescene.features.chat.service.messages :as messages]
            [taoensso.timbre :as log]))

(defn reset-mcp-session!
  "Reset the MCP session for a specific auth token."
  [auth-token]
  (mcp/reset-mcp-session! auth-token))

(defn list-prompts
  "List available prompts from all MCP servers.
   
   Arguments:
   - context: Map with :auth-token, :instance-url, and optionally :mcp-config, :servers
   
   Returns a vector of prompt maps with :name, :title, :description, :arguments"
  [{:keys [auth-token instance-url mcp-config servers]}]
  (:prompts (mcp/list-prompts {:auth-token auth-token
                               :instance-url instance-url
                               :mcp-config mcp-config
                               :servers servers})))

(defn get-prompt
  "Get a specific prompt with arguments applied.
   
   Arguments:
   - context: Map with :auth-token, :instance-url, :prompt-name, :arguments, and optionally :mcp-config, :servers
   
   Returns:
   {:success true/false
    :result {:description string :messages [...]} or
    :error string}"
  [{:keys [auth-token instance-url prompt-name arguments mcp-config servers]}]
  (mcp/get-prompt {:auth-token auth-token
                   :instance-url instance-url
                   :prompt-name prompt-name
                   :arguments arguments
                   :mcp-config mcp-config
                   :servers servers}))

(defn- format-mcp-warning
  "Formats MCP connection errors into a warning string for the user."
  [errors]
  (when (seq errors)
    (str "⚠️ **Warning:** " (first errors)
         (when (> (count errors) 1)
           (str " (and " (dec (count errors)) " more connection issues)")))))

(defn- prepend-warning-to-response
  "Prepends a warning message to the response content if there are MCP errors."
  [result warning]
  (if (and warning (:success result))
    (update-in result [:response :content]
               (fn [content]
                 (str warning "\n\n---\n\n" content)))
    result))

(defn chat-with-tools
  "Main chat function with tool execution loop.
   
   Arguments:
   - llm-config: Map with :provider, :model, :api-key, :base-url
   - context: Map with :messages, :project-id, :auth-token, :instance-url, and optionally :mcp-config
   
   Returns:
   {:success true/false
    :response {:role :assistant :content string}
    :thinking string (optional)
    :usage {:prompt-tokens int :completion-tokens int}
    :iterations int
    :error string (only if success is false)}"
  [llm-config {:keys [messages auth-token instance-url mcp-config servers] :as context}]
  (let [{:keys [tools errors]} (mcp/list-tools {:auth-token auth-token 
                                                 :instance-url instance-url
                                                 :mcp-config mcp-config
                                                 :servers servers})
        warning (format-mcp-warning errors)
        prepared-messages (messages/prepare-with-tools messages tools)]
    (log/infof "Starting chat with %d tools" (count tools))
    (when (seq errors)
      (log/warnf "MCP connection errors: %s" (pr-str errors)))
    (-> (loop/run llm-config (assoc context :messages prepared-messages))
        (prepend-warning-to-response warning))))

