(ns codescene.features.util.set
  "Handy utilities built on top of clojure.set and/or extending set-related operations.
  The suggested alias for this ns is `uset`.

  `full-join` provides a more generic 'FULL OUTER JOIN' operation
  but only allows a single join-key to be used.
  `left-join` is less generic, but accepts a map of join keys.

  Resources:
  - https://stackoverflow.com/questions/13009939/outer-join-in-clojure
  - https://statcompute.wordpress.com/2018/04/08/inner-and-outer-joins-in-clojure/
  - ** https://gist.github.com/ataggart/d00cfde376476c9e495882a60deba904 **"
  (:require [clojure.set :as set]))


(defn full-join
  "Similar `clojure.set/join` but with FULL OUTER JOIN semantics, that is
  preserving also the records that are only in one of the sets.
  Accepts only a single join key (unlike `set/join` which accepts a map of keys)"
  [s1 s2 join-key]
  (->> (group-by join-key (concat s1 s2))
       vals
       (map #(apply merge %))
       set))


(defn left-join
  "Returns the rel corresponding to the naturaljoin of two relations (sets).
  Unlike `clojure.set/join` always expects an additional keymap (that's what we want in most cases)
  joining on the corresponding keys.

  Adapted from https://gist.github.com/ataggart/d00cfde376476c9e495882a60deba904"
  ([xrel yrel km]
   (let [yrel-idx (set/index yrel (vals km))]
     (reduce (fn [ret x]
               (if-let [found-in-yrel (get yrel-idx (set/rename-keys (select-keys x (keys km)) km))]
                 (reduce #(conj %1 (merge x %2)) ret found-in-yrel)
                 ;; the element is not in `yrel` so just include it as is - no merging needed
                 (conj ret x)))
             #{}
             xrel))))

;; see also comment block at the end of the `clojure.set` ns
(comment

  (def xs #{{:a 11 :b 1 :c 1 :d 4}
            {:a 2 :b 12 :c 2 :d 6}
            {:a 3 :b 3 :c 3 :d 8 :f 42}
            {:a 4 :b 4 :c 4 :d 4}})

  (def ys #{{:a 11 :b 11 :c 11 :e 5}
            {:a 12 :b 11 :c 12 :e 3}
            {:a 3 :b 3 :c 3 :e 7 }})

  (left-join xs (set/rename ys {:a :aa :c :yc}) {:a :aa})
  ;; => #{{:a 4, :b 4, :c 4, :d 4} ; this is only in `xs`
  ;;      {:a 11, :b 11, :c 1, :d 4, :e 5, :aa 11, :yc 11} ; value of `b` is overriden by `ys` (it's 1 in `xs`)
  ;;      {:a 3, :b 3, :c 3, :d 8, :f 42, :e 7, :aa 3, :yc 3}
  ;;      {:a 2, :b 12, :c 2, :d 6}}


,)
