(ns codescene.features.util.log
  (:require [clojure.string :as string]
            [taoensso.timbre :as log]))

(def secrets-log-patterns
  "All patterns matching potentially sensitive data in logs that should be replaced by `secret-replacement-str`.

  These patterns should always follow the format '(Whatever is the matching prefix) MYSCRET'.
  - note that the parens are required!
  This will match the log message 'Whatever is the matching prefix MYSCRET'
  and that message will be replaced with 'Whatever is the matching prefix ***'"
  [;; Basic Auth
   #"([Aa]uthorization:\s+\w+\s+)\S+"
   #"(:basic-auth\s).*"
   ;; Private token
   #"(PRIVATE-TOKEN:\s+)[^\s]+"
   ;; OAuth related secrets
   #"([:\"](?:access_token|access-token|refresh_token|refresh-token)(?:\":)? \")[^\"]+" ;; OAuth access/refresh tokens as used internally in Clojure maps or returned by OAuth provider "access_token" API
   #"(client_secret=)[^\s&]+" ;; OAuth app client secret sent in the request form body param
   #"(:client-secret \")[^\"]+" ;; OAuth app client secret as stored in the CodeScene auth provider configuration
   ;; trello-provider credentials
   #"(:token\s)\S+"
   #"(token=)[^\s&]+" ; yes, they require tokens to be set in query string!
   #"(https?://)[^@\s]++@+"
   ])

(def secret-replacement "***")

(defn- mask-secrets [secrets-patterns replacement log-message]
  (when log-message
    (reduce
      (fn mask-pattern [msg pattern]
        ;; the original prefix ($1) plus the secret replaced with '***'
        (string/replace msg pattern (str "$1" replacement)))
      log-message
      secrets-patterns)))

(def mask-secrets-in-log (partial mask-secrets secrets-log-patterns secret-replacement))

(defmacro time-expr
  "As `clojure.core/time` but returns a tuple [return-value elapsed-time-in-millis]
  and doesn't print the elapsed time."
  [expr]
  `(let [start# (System/nanoTime)
         ret# ~expr
         elapsed-msecs# (/ (double (- (System/nanoTime) start#)) 1000000.0)]
     [ret# elapsed-msecs#]))
#_(second (time-expr (mapv inc (range 1000000))))
;; => 29.489776

(defmacro log-time
  "Wraps `expr` with 2 log statements with `msg` being the log message.
  One message is logged before executing the expression
  and one after the expression has been executed - the second one
  also contains the duration of the whole execution.
  The message can contain placeholders to be replaced by `msg-args`."
  [expr level msg & msg-args]
  `(let [_# (log/logf ~level ~msg ~@msg-args)
         [ret# time#] (time-expr ~expr)]
     (log/logf ~level (str ~msg " FINISHED [Elapsed time: %s msecs]") ~@msg-args time#)
     ret#))
#_(take 10 (log-time (mapv inc (range 1000000))
                   :info
                   "log %s this %s"
                   "some" "extra"))

(defmacro log-excessive-time
  "Invokes the log expression if execution of expr exceeds the time limit (default 2 seconds).
  Elapsed time in seconds is available to log expression as a local variable '%t'."
  [expr log-expression & time-limit-seconds]
  `(let [[ret# time#] (time-expr ~expr)
         ~'%t (/ time# 1000.0)]
     (when (> ~'%t ~(or (first time-limit-seconds) 2.0))
       ~log-expression)
     ret#))
#_(take 10 (log-excessive-time (mapv inc (range 1000000))
                             (log/infof "took too long: %s" %t)
                             0.001))

(defmacro log-excessive-as->
  "Similar to `as->`, but for each `expr` a `log-expr` is supplied,
  using `log-excessive-time`.

  (log-excessive-as-> 2 <<marker>>
    (the-first-value b)
    (log/infof \"The first value took %s seconds\" %t)

    (the-second-value <<marker>>)
    (log/infof \"The second value took %s seconds\" %t))
  "
  [time-limit marker expr log-expr & expr-log-expr-pairs]
  (let [time-lim-gs (gensym "time-limit")]
    `(let [~time-lim-gs ~time-limit]
       (as-> (log-excessive-time ~expr ~log-expr ~time-lim-gs) ~marker
         ~@(map (fn [[expr log-expr]]
                  `(log-excessive-time ~expr ~log-expr ~time-lim-gs))
                (partition 2 expr-log-expr-pairs))))))


;;; We get some extra goodies by preserving also the client stacktrace
;;; See https://www.nurkiewicz.com/2014/11/executorservice-10-tips-and-tricks.html
(defn logging-future+* [file line body]
  `(let [client-stack-trace# (Exception. "Client stack trace")]
     (future
       (try ~@body
            (catch Throwable e#
              (log/error e# "Unhandled exception at:"
                         ~file "line:" ~line
                         "on thread:"
                         (.getName (Thread/currentThread)))
              (log/error client-stack-trace# "client stack trace:"))))))

(defmacro logging-future+
  "Logs any error that occurs during the execution of given body in a `future`
  *including* the client stack trace at the time of submitting the future for execution."
  [& body]
  (logging-future+* *file* (:line (meta &form)) body))
