(ns codescene.features.util.item-filter
  (:require [clojure.edn :as edn]
            [clojure.string :as str]
            [slingshot.slingshot :refer [throw+]]))
  
(defn- ->num-pred [f op s]
  (let [value (edn/read-string s)
        pred #(f % value)]
    (if (number? value)
      (fn [field-value]
        (and field-value (number? field-value) (pred field-value)))
      (throw+ {:type :invalid-analyses-request :message (format "The %s operator is only supported for numbers" op)}))))

(defn- ->str-pred [f s]
  (let [value (str/lower-case s)
        pred #(f (str/lower-case %) value)]
    (fn [field-value]
      (and (some? field-value) (pred (str field-value))))))

;; Validators operate on the collected field-values (a sequence).
;; "not-empty" / "empty" check the collection itself
;; "some" / "none" check if any/all values are nil (useful for nested paths)
(def ^:private validators-map
  {"not-empty" seq
   "empty" empty?
   "some" #(some some? %)
   "none" #(every? nil? %)})

(defn- ->validator-pred [s]
  (or (get validators-map s)
      (throw+ {:type :invalid-analyses-request
               :message (format "The %s validator is not supported." s)})))

;; A map from operator symbols to a fn taking a comparison value and creating a predicate
(def operator->pred-maker
  {">" #(->num-pred > ">" %)
   "<" #(->num-pred < "<" %)
   "=" #(->num-pred = "=" %)
   ":" #(->str-pred = %)
   "~" #(->str-pred str/includes? %)
   "^" ->validator-pred})

(def clause-pattern
  ;; match strings like 'a:b' or a:"b" with ':' being any of the supported operators
  ;; and capture (FIELD, OPERATOR, QUOTED-VALUE VALUE) in subgroups
  (let [operators (->> (keys operator->pred-maker)
                       (map #(str "\\" %))
                       (str/join "|"))]
    (re-pattern (format "^\\s*([\\w\\.]+)(%s)(?:\"([^\"]+)\"|([^\\s]+))\\s*" operators))))

(defn- field-values
  "A rather hackyish fn for getting nested field values in a collection
   This means that in
   {:a [{:b 1}
        {:b 2}] 
    :c {:d 3} 
    :d 4})}
   the results will be
   [:a :b] => [1 2]
   [:c :d] => [3]
   [:d] => [4]"
  [m ks]
  (reduce (fn [acc kw]
            (mapcat
             #(let [v (kw %)]
                (if (and (coll? v) (not (map? v)))
                  v
                  [v]))
             acc))
          [m]
          ks))

(defn- clause-match->pred
  [[_ field operator quoted-value value]]
  (let [pred ((get operator->pred-maker operator) (or quoted-value value))
        ks (->> (str/split field #"\.") (map keyword))]
    (fn [item]
      (let [field-values (field-values item ks)]
        (if (= operator "^")
          (pred field-values)
          (some pred field-values))))))

(defn- parse-clauses 
  "Parse the clauses present in the query string and return a list of predicates"
  [s]
  (loop [preds []
         remaining s]
    (if-let [match (re-find clause-pattern remaining)]
      (recur (conj preds (clause-match->pred match))
             (subs remaining (count (first match))))
      (if (empty? remaining)
        preds
        (throw+ {:type :invalid-analyses-request :message (format "The clause '%s' is invalid" remaining)})))))

(defn parse-filter*
  [s]
  (let [preds (parse-clauses s)]
    (apply every-pred preds)))
  
(defn parse-filter
  "Parse a query string and return a predicate that can be used for filtering items"
  [filter]
  (if (seq filter)
    (comp boolean (parse-filter* filter))
    (constantly true)))
