(ns codescene.features.time-series.time-series-db
  (:require
   [codescene.util.json :as json]
   [hugsql.core :as hugsql]
   [codescene.features.util.string :refer [declob]]))


(hugsql/def-db-fns "codescene/features/time_series/time-series.sql")
(declare insert-time-series-entry)
(declare get-all-time-series-entries-for-project)
(declare get-all-time-series-entries-for-project-until)
(declare get-all-time-series-entries-for-project-by-range)
(declare get-most-recent-time-series-entry-for-project)

(def time-series-types #{"code-coverage" "project-status"})

(defn- entry->insertable
  [{:keys [project-id timestamp payload]}]
  {:project_id project-id
   :timestamp timestamp
   :time_series_json (json/generate-string payload)})

(defn insert-entry
  "The `timestamp` field needs to be a real timestamp, eg. (tcoerce/to-timestamp...)"
  [db-spec time-series-type entry]
  (when-not (contains? time-series-types time-series-type)
    (throw (ex-info "Cannot insert unknown time series type." {:project-id (:project-id entry)
                                                               :unknown-type time-series-type})))
  (insert-time-series-entry db-spec (-> entry
                                        entry->insertable
                                        (assoc :time_series_type time-series-type))))



(defn get-complete-time-series
  [db-spec time-series-type project-id _no-timestamps]
  (->> (get-all-time-series-entries-for-project db-spec {:project_id project-id
                                                         :time_series_type time-series-type})
       (map (comp declob :time_series_json))))


(defn get-time-series-up-to
  [db-spec time-series-type project-id {:keys [end-timestamp]}]
  (->> (get-all-time-series-entries-for-project-until db-spec {:project_id project-id
                                                               :time_series_type time-series-type
                                                               :end_timestamp end-timestamp})
       (map (comp declob :time_series_json))))


(defn get-time-series-between
  [db-spec time-series-type project-id {:keys [start-timestamp end-timestamp]}]
  (->> (get-all-time-series-entries-for-project-by-range db-spec {:project_id project-id
                                                                 :time_series_type time-series-type
                                                                 :start_timestamp start-timestamp
                                                                 :end_timestamp end-timestamp})
       (map (comp declob :time_series_json))))

(defn- neither [a b] (and (not a) (not b)))

(defn- time-series-dispatch-on-range
  [{:keys [start-timestamp end-timestamp] :as _range}]
  (cond
    (neither start-timestamp end-timestamp) :neither
    (and start-timestamp end-timestamp) :both
    end-timestamp :end-only
    ;; There's currently no need for an "after x" variation here. Could be added though.
    :else :invalid))                    

(defmulti get-time-series-for-range (fn [_db-spec _time-series-type _project-id range]
                                      (time-series-dispatch-on-range range)))

(defmethod get-time-series-for-range :neither [db-spec time-series-type project-id _range]
  (get-complete-time-series db-spec time-series-type project-id {}))

(defmethod get-time-series-for-range :end-only [db-spec time-series-type project-id range]
  (get-time-series-up-to db-spec time-series-type project-id range))

(defmethod get-time-series-for-range :both [db-spec time-series-type project-id range]
  (get-time-series-between db-spec time-series-type project-id range))

(defmethod get-time-series-for-range :invalid [_db-spec time-series-type project-id range]
  (throw (ex-info "Invalid timestamp limits for time series query"
                  {:time-series-type time-series-type
                   :project-id project-id
                   :limits range})))

(defn get-most-recent-time-series
  "Fetches the most recent time series entry of the given type for a project.
   Returns nil if no entries exist.
   
   The result is parsed from JSON into a Clojure map with keyword keys."
  [db-spec time-series-type project-id]
  (when-not (contains? time-series-types time-series-type)
    (throw (ex-info "Unknown time series type." {:project-id project-id
                                                :unknown-type time-series-type})))
  (when-let [result (get-most-recent-time-series-entry-for-project 
                      db-spec 
                      {:project_id project-id
                       :time_series_type time-series-type})]
    (-> result
        :time_series_json
        declob
        (json/parse-string true))))
; (ns-unmap *ns* 'get-time-series-for-range)
