(ns codescene.features.scheduling.persistent
  "Persistent scheduler backed by database tables (look for 'scheduled_tasks')."
  (:require [taoensso.timbre :as log])
  (:import (clojure.lang PersistentArrayMap)
           (com.github.kagkarlsson.scheduler Scheduler)
           (com.github.kagkarlsson.scheduler.serializer Serializer)
           (com.github.kagkarlsson.scheduler.task SchedulableInstance$Builder VoidExecutionHandler)
           (com.github.kagkarlsson.scheduler.task.helper Tasks)
           (java.time Duration Instant)
           (java.util List)
           (javax.sql DataSource)))

(defonce ^:dynamic *persistent-sched* nil)

(defn scheduler
  "Returns the persistent scheduler.
  Prefer this to directly accessing `*persistent-sched*`."
  []
  (assert *persistent-sched* "Scheduler has not been initialized; run init")
  *persistent-sched*)

(comment
  (def edn-serializer
    (reify Serializer
      (serialize [this data])
      (deserialize [this clazz serialized-data]))))

(defn schedule-task
  "Schedules a one-time task. f-sym is qualified function symbol. delay is java.time.Duration"
  ([f-sym data delay]
   (schedule-task f-sym data delay (str (random-uuid))))
  ([f-sym data delay id]
   (.schedule ^Scheduler *persistent-sched*
              (-> (SchedulableInstance$Builder. "one-time" id)
                  (.data (assoc data ::f f-sym))
                  (.scheduledTo (cond-> (Instant/now) delay (.plus delay)))))))

(defn make-scheduler [db-spec]
  (-> (Scheduler/create ^DataSource (:datasource db-spec)
                        ^List [(.execute (Tasks/oneTime "one-time" PersistentArrayMap)
                                         (reify
                                           VoidExecutionHandler
                                           (execute [this inst context]
                                             (let [data (.getData inst)
                                                   f @(requiring-resolve (::f data))]
                                               (f (dissoc data ::f)
                                                  {:task-instance inst
                                                   :task-context context})))))])
      (.pollingInterval (Duration/ofSeconds 2))
      (.threads 5)
      (.registerShutdownHook)
      (.build)))

(defn init
  "Creates the scheduler, it's available to queue up jobs, but it doesn't start the
  job loop. This can be done fairly early in startup process so this service is available
  to other modules during startup."
  [db-spec]
  (log/info "Creating db-scheduler persistent scheduler")
  (alter-var-root #'*persistent-sched*
                  (constantly (make-scheduler db-spec))))

(defn start
  "Starts the scheduler. This should be done late in startup so that any other modules
  that are used by the jobs are initialized."
  []
  (log/info "Starting db-scheduler persistent scheduler")
  (.start *persistent-sched*))

(defn destroy []
  (log/info "Stopping Db-Scheduler persistent scheduler")
  (when *persistent-sched*
    (.stop ^Scheduler *persistent-sched*)
    (alter-var-root #'*persistent-sched* (constantly nil))))

(defmacro with-test-scheduler [db-spec & body]
  `(binding [*persistent-sched* (make-scheduler ~db-spec)]
     (.start *persistent-sched*)
     (try
       ~@body
       (finally
         (.stop ^Scheduler *persistent-sched*)))))
