(ns codescene.features.pm-data.trello.trello-cache
  "Wraps some fetcher fns to make them cacheable"
  (:require [codescene.features.pm-data.trello.trello-fetcher :as f]
            [codescene.features.pm-data.cache :as cache]))

(def fetch-cards-version 2) ;; Increment this if the data returned changes withot the cache key changing

(defn fetch-cards-key-fn
  [_ _ board-id {:keys [supported-work-types] :as _provider-def}]
  [fetch-cards-version board-id supported-work-types])

(def ^:private fetch-cards-cacheable (cache/make-cacheable f/fetch-cards))
(cache/memo #'fetch-cards-cacheable fetch-cards-key-fn)
  
(defn fetch-actions-key-fn
  [_ _ board-id]
  [board-id])

(def ^:private fetch-actions-cacheable (cache/make-cacheable f/fetch-actions))
(cache/memo #'fetch-actions-cacheable fetch-actions-key-fn)

(defn ->actions-lookup
  [actions]
  (reduce (fn [acc {:keys [action card] :as a}]
            (update acc card
                    (fn [{:keys [closed moved] :or {closed [] moved []} :as m}]
                      (case action
                        :moved  (assoc m :moved (conj moved a))
                        :closed (assoc m :closed (conj closed a))
                        m))))
          {}
          actions))
  
(defn ->issue
  [issue {:keys [moved closed] :as _actions}]
  (assoc issue
         :transitions (->> moved
                           (sort-by :date)
                           (mapv (juxt :state :date)))
         :closed      (->> closed
                           (sort-by :date)
                           last
                           :date)))
  
(defn- fetch-actions
  [since api-client board-id]
  (cache/combine-with-cache fetch-actions-cacheable fetch-actions-key-fn
                            since api-client board-id))

(defn- fetch-cards
  [since api-client board-id provider-def]
  (cache/combine-with-cache fetch-cards-cacheable fetch-cards-key-fn
                            since api-client board-id provider-def))

(defn fetch-issues
  [since api-client project provider-def]
  (let [boards    (f/fetch-boards api-client)
        board-id  (get boards project)
        cards     (fetch-cards since api-client board-id provider-def)
        ->actions (-> (fetch-actions since api-client board-id)
                      ->actions-lookup)]
    (map (fn [{:keys [id] :as issue}]
           (->issue issue (->actions id)))
         cards)))


(comment
  (def api-client (codescene.features.pm-data.trello.trello-api/trello-auth
                   (System/getenv "TRELLO_API_KEY")
                   (System/getenv "TRELLO_API_TOKEN")))
  (def board-name "CodeScene")
  (def board-id "6045ea5748782b336a1feafb")

  (let [since (evolutionary-metrics.trends.dates/string->date "2022-03-01")]
    (codescene.cache.core/with-file-cache "." "." :pm-data
      (-> (fetch-actions since api-client board-id)
          ->actions-lookup
          (get "1363"))))
  (let [since (evolutionary-metrics.trends.dates/string->date "2021-03-01")]
    (codescene.cache.core/with-file-cache "." "." :pm-data
      {:issues (count (fetch-cards since api-client board-id {}))}))
  (let [since (evolutionary-metrics.trends.dates/string->date "2022-02-01")]
    (codescene.cache.core/with-file-cache "." "." :pm-data
      (filter #(= "1363" (:id %)) (fetch-issues since api-client board-name {}))))
  )