(ns codescene.features.pm-data.gitlab.fetcher
  "Uses the GitLab API for fetching data.
   This includes combining and transforming data to internal pm-data format, and in some cases catching and reporting errors."
  (:require [clj-time.coerce :as c]
            [clj-time.core :as t]
            [codescene.cache.core :as cache]
            [codescene.url.url-utils :as url]
            [taoensso.timbre :as log]
            [codescene.features.repository-provider.gitlab.api :as api]
            [codescene.pm-data.provider-common :as common]
            [evolutionary-metrics.trends.dates :as dates]))

(defn- pr->result [authed-client repo-info {:keys [description title iid source_branch merge_commit_sha updated_at] :as _pr}]
  {:id (str iid)
   :title title
   :branch source_branch
   :text description
   :merge-commit merge_commit_sha
   :commits (api/get-merge-request-commits authed-client repo-info iid)
   :updated (dates/timestamp->date updated_at)})

(defn fetch-merge-requests
  "Fetches merged pull-requests from the remote GitLab API. Throws when the API calls fail."
  [since authed-client repo-info]
  (log/infof "Fetch merge requests from GitLab since %s" (or since "-"))
  (let [search-options {:since since
                        :state "merged"}]
    (->> (api/get-merge-requests authed-client repo-info search-options)
         (mapv (partial pr->result authed-client repo-info)))))

(defn fetch-labels
  [authed-client repo-urls]
  (->> repo-urls
       (keep #(url/repo-url->repo-info* % :gitlab false))
       (mapcat #(api/get-labels authed-client %))
       (map :name)
       distinct
       (mapv common/->name-and-key)))

(defn fetch-board-labels
  [authed-client repo-urls]
  (->> repo-urls
       (keep #(url/repo-url->repo-info* % :gitlab false))
       (mapcat #(api/get-boards authed-client %))
       (mapcat :lists)
       (map (comp :name :label))
       distinct
       (mapv common/->name-and-key)))

(defn- non-zero-time [t] (when-not (zero? t) t))

(defn- issue-event->transition [{:keys [action label created_at]}]
  (when (and (= "add" action) label)
    [(:name label) (common/drop-ms-from-time-string created_at)]))

(defn- fetch-transitions
  [authed-client repo-info iid]
  (->> (api/get-issue-label-events authed-client repo-info iid)
       (keep issue-event->transition)
       vec))

(defn- issue->result
  [provider-def iid->transitions issue]
  (let [{:keys [created_at closed_at time_stats labels web_url iid references title]} issue
        {:keys [supported-work-types]} provider-def]
    {:id          (:full references)
     :href        web_url
     :task-name   title
     :created     (common/drop-ms-from-time-string created_at)
     :closed      (common/drop-ms-from-time-string closed_at)
     ;:updated     (or closed_at created_at)
     :status nil
     :cost 0
     :time-spent (:total_time_spent time_stats)
     :work-types (filterv (set supported-work-types) labels)
     :transitions (cond-> (conj (iid->transitions iid)
                                ["Opened" (common/drop-ms-from-time-string created_at)])
                    closed_at (conj ["Closed" (common/drop-ms-from-time-string closed_at)]))}))

(defn fetch-issues
  [since authed-client repo-url provider-def]
  (if-let [repo-info (url/repo-url->repo-info* repo-url :gitlab false)]
    (let [iid->transitions (partial fetch-transitions authed-client repo-info)]
      (->> (api/get-issues authed-client repo-info since)
           (map (partial issue->result provider-def iid->transitions))))
    []))

(cache/memo-scoped #'fetch-labels)
(cache/memo-scoped #'fetch-board-labels)

(comment
  (def authed-client (System/getenv "GITLAB_TOKEN"))
  (def repo-info (codescene.url.url-utils/repo-url->repo-info "git@gitlab.com:empear/analysis-target.git" :gitlab false))
  (fetch-merge-requests (dates/string->date "2021-07-04") authed-client repo-info))
  
