(ns codescene.features.pm-data.azure.azure-provider
  "A PM Data Provider for fetching data from Azure.
   Uses a cache for fetching data from Azure. 
   This means that a single instance of the Provider will only ever fetch data from Azure once,
   and a new instance (with a new cache) is needed for fetching updated data."
  (:require [codescene.pm-data.pm-data-provider :refer [PmDataProvider]]
            [codescene.features.pm-data.azure.azure-fetcher :as f]
            [codescene.features.pm-data.azure.azure-cache :as ac]
            [codescene.features.pm-data.azure.azure-ticket-id-extractor :as x]
            [codescene.features.pm-data.pm-data-configuration :refer [PmDataConfiguration]]
            [codescene.features.pm-data.pm-data-provider :refer [-create-pm-data-provider -uses-supported-work-types]]
            [codescene.features.util.log :as u.log]
            [codescene.url.url-utils :as url]
            [medley.core :as m]
            [taoensso.timbre :as log]))

(defn- host->api-url [host api-url]
  (cond
    (not-empty api-url) api-url
    (not-empty host) (str "https://" host)
    :else "https://dev.azure.com"))

(defn- with-scoped-ids [project-parts issues]
  (let [->scoped-id (x/->ticket-id-scoper project-parts)]
    (map #(update % :id ->scoped-id) issues)))

(defn- with-href [api-url {:keys [owner project] :as _project-parts} issues]
  (let [->href #(format "%s/%s/%s/_workitems/edit/%s" api-url owner project %)]
    (map #(assoc % :href (->href (:id %)))
         issues)))

(defn- owner-from [repo-urls]
  (-> repo-urls first url/azure-url->parts* :owner))

(defn valid-azure-url? 
  "Checks whether the given repo-url is a valid Azure DevOps url from which an owner can be extracted."
  [repo-url]
  (boolean (-> repo-url url/azure-url->parts* :owner)))

(defn- projects-from
  [repo-urls]
  (->> repo-urls
       (keep url/azure-url->parts*)
       (m/distinct-by :project)))

(defn get-project-parts-coll 
  "Projects are either selected by the user (if external-project-ids is set)
   or extracted from repo urls"
  [{:keys [organization external-project-ids api-url] :as _provider-def} repo-urls]
  (if (seq external-project-ids)
    (if-let [org (or organization (owner-from repo-urls))]
      (map (fn [project]
             {:host api-url
              :owner org
              :project project})
           external-project-ids)
      [])
    (projects-from repo-urls)))

(defn- make-ticket-id-extractor [provider-def repo-url pm-data-context]
  (let [project-parts-coll (get-project-parts-coll provider-def [repo-url])]
    (x/make-ticket-id-extractor provider-def repo-url project-parts-coll pm-data-context)))

(defn fetch-project [{:keys [api-client api-url] :as provider-def} {:keys [host project] :as project-parts} since]
  (let [api-url (host->api-url host api-url)
        issues (->> (ac/fetch-issues since api-client project-parts provider-def)
                    (with-href api-url project-parts)
                    (with-scoped-ids project-parts))]
    (log/infof "Import %d issues from Azure project %s" (count issues) project)
    issues))

(defn- get-pm-data
  [provider-def {:keys [repo-urls since] :as _pm-data-context} ]
  (let [issues (->> (get-project-parts-coll provider-def repo-urls)
                    (mapcat #(fetch-project provider-def % since))
                    (into []))]
    {:issues issues
     :provider-def provider-def}))

(defn- project-filter
  "Create a predicate that filters out projects that are not among the repo-urls,
   unless org has been configured by user (because repos are not on Azure)"
  [{:keys [organization] :as _provider-def} {:keys [repo-urls] :as pm-data-context}]
  (let [project-names (->> (projects-from repo-urls) (map :project) set)]
    (if (or (seq organization) (empty? project-names))
      ;; if org is selected by user -> keep all projects...
      (constantly true)
      ;; ...but if not -> keep the ones that match projects from repo urls
      (fn [{:keys [name]}]
        (contains? project-names name)))))

(defn- get-configuration-data
  [{:keys [api-client organization] :as provider-def}
   {:keys [repo-urls] :as pm-data-context}]
  (u.log/log-time
      (let [owner  (or (not-empty organization) (owner-from repo-urls))
            fields (future (f/fetch-number-fields api-client owner))
            ;; not a future b/c result is required for next call
            projects (f/fetch-projects api-client owner)
            project-keys (->> projects
                             (filter (project-filter provider-def pm-data-context))
                             (mapv :key))
            ;; not a future b/c `fetch-work-item-types*` has its own threads
            transitions (future (f/fetch-work-item-states api-client owner project-keys))
            work-types (f/fetch-work-item-types api-client owner project-keys)]
        (m/assoc-some {:work-types work-types
                       :fields  @fields
                       :transitions @transitions}
                      ;; Projects will be set if the repos are not on azure - because then they have to be selected by the user
                      :projects (when (seq owner) projects)))
      :info
      "PM Data: Fetching Azure Devops configuration data"))

(defn- validate-settings
  [{:keys [api-client organization] :as _provider-def} {:keys [repo-urls] :as _pm-data-context}]
  (let [owner (or (not-empty organization) (owner-from repo-urls))]
    (f/fetch-projects api-client owner)
    nil))

(deftype AzureProvider [provider-def pm-data-context]
  PmDataProvider
  (-make-ticket-id-extractor [_this repo] (make-ticket-id-extractor provider-def repo pm-data-context))
  (-get-pm-data [_this] (get-pm-data provider-def pm-data-context))
  PmDataConfiguration
  (-get-configuration-data [_this] (get-configuration-data provider-def pm-data-context))
  (-validate-settings [_this] (validate-settings provider-def pm-data-context)))

(defmethod -create-pm-data-provider "azure"
  [provider-def pm-data-context]
  (->AzureProvider provider-def pm-data-context))

(defmethod -uses-supported-work-types "azure"
  [_provider-def]
  false)

(comment
  (def repo-url "git@ssh.dev.azure.com:v3/empear/SmartHotel360/PublicWeb")
  (def repo-url "git@github.com:empear-analytics/analysis-target.git")
  (def api-client ["" (System/getenv "AZURE_TOKEN")])
  (def provider-def {:map-commits-to-pull-request-issues "on"
                     :map-subtasks-to-parent-issues "on"
                     :cost-field "Microsoft.VSTS.Scheduling.Effort"
                     :cost-unit "points"
                     :external-project-ids ["650a76c2-d1f4-4085-a6b8-d87bf4e5a6bd"]
                     :rename-work-types [["Task" "Hohoho"]]
                     :defect-and-failure-labels ["Bug"]
                     :work-in-progress-transition-names "In Progress"
                     :work-done-transition-names "Done"
                     :api-client api-client})
  (def context {:repo-urls [repo-url]})

  (get-configuration-data provider-def context)
  (validate-settings provider-def context)
  (get-pm-data provider-def context nil)
  )
