(ns codescene.features.components.analysis
  (:require [clojure.spec.alpha :as s]
            [codescene.features.spec.commons :as commons-spec])
  (:import (java.sql Timestamp)))

(s/def ::path-fn fn?)
(s/def ::dir string?)
(s/def ::description string?)
(s/def ::category string?)
(s/def ::explanation string?)
(s/def ::linkable-name string?)
(s/def ::analysistime #(instance? Timestamp %))

(s/def ::early-warning (s/keys :req-un [::commons-spec/name ::description ::category ::explanation ::linkable-name]))

;; Analysis summary specs
(s/def :analysis.summary/now (s/nilable double?))
(s/def :analysis.summary/month (s/nilable double?))
(s/def :analysis.summary/year (s/nilable double?))
(s/def :analysis.summary/change_month (s/nilable double?))
(s/def :analysis.summary/change_year (s/nilable double?))
(s/def :analysis.summary/total (s/nilable int?))
(s/def :analysis.summary/active (s/nilable int?))
(s/def :analysis.summary/code-health (s/keys :req-un [:analysis.summary/now :analysis.summary/month :analysis.summary/year
                                                      :analysis.summary/change_month :analysis.summary/change_year]))
(s/def :analysis.summary/hotspot-code-health (s/keys :req-un [:analysis.summary/now :analysis.summary/month :analysis.summary/year
                                                              :analysis.summary/change_month :analysis.summary/change_year]))
(s/def :analysis.summary/authors (s/keys :req-un [:analysis.summary/total :analysis.summary/active]))
;; Delivery summary specs (optional - only present when PM integration is configured)
(s/def :analysis.summary/unplanned-work-percent (s/nilable number?))
(s/def :analysis.summary/avg-development-time-minutes (s/nilable number?))
(s/def :analysis.summary/delivery (s/keys :opt-un [:analysis.summary/unplanned-work-percent
                                                   :analysis.summary/avg-development-time-minutes]))
;; Code coverage specs (optional - only present when code coverage data is available)
(s/def :analysis.summary/line-coverage-percent (s/nilable double?))
(s/def :analysis.summary/code-coverage (s/keys :opt-un [:analysis.summary/line-coverage-percent]))
(s/def ::analysis-summary (s/keys :req-un [:analysis.summary/code-health :analysis.summary/hotspot-code-health :analysis.summary/authors]
                                  :opt-un [:analysis.summary/delivery :analysis.summary/code-coverage]))

(defn build-summary
  "Pure function that assembles the final summary from its components.
   Takes a map with :base-summary and optional :delivery and :code-coverage.
   Shared between cloud and on-prem implementations."
  [{:keys [base-summary delivery code-coverage]}]
  (cond-> base-summary
    delivery (assoc :delivery delivery)
    code-coverage (assoc :code-coverage code-coverage)))

(defprotocol AnalysisComponent
  (-latest-successful-analysis [this project-id])
  (-latest-successful-analysis-summary [this project-id])
  (-early-warnings [this project-id analysis-id])
  (-local-analysis-result-path [this project-id analysis-id])
  (-all-analyses-in-project-filtered [this project-id filter-fn])
  (-analysis-by-id [this project-id analysis-id])
  (-latest-couplings [this project-id]))

(s/fdef latest-successful-analysis
  :args (s/cat :this some?
               :project-id ::commons-spec/id)
  :ret (s/nilable (s/keys :opt-un [::commons-spec/id ::dir ::commons-spec/name])))
(defn latest-successful-analysis
  "Returns analysis results for the last successful analysis if any."
  [this project-id]
  (-latest-successful-analysis this project-id))

(s/fdef latest-successful-analysis-summary
  :args (s/cat :this some?
               :project-id ::commons-spec/id)
  :ret (s/nilable ::analysis-summary))
(defn latest-successful-analysis-summary
  "Returns the analysis summary of the last successful analysis."
  [this project-id]
  (-latest-successful-analysis-summary this project-id))

(s/fdef local-analysis-result-path
        :args (s/cat :this some?
                     :project-id ::commons-spec/id
                     :analysis-id ::commons-spec/id)
        :ret ::path-fn)
(defn local-analysis-result-path
  "Makes analysis results available in a local file system, returns a function which take one parameter,
  the path to analysis resource file, and build the full filename path"
  [this project-id analysis-id]
  (-local-analysis-result-path this project-id analysis-id))

(s/fdef early-warnings
        :args (s/cat :this some?
                     :project-id ::commons-spec/id
                     :analysis-id ::commons-spec/id)
        :ret (s/coll-of ::early-warning))
(defn early-warnings
  "Returns early warnings for analysis id"
  [this project-id analysis-id]
  (-early-warnings this project-id analysis-id))

(s/fdef all-analyses-in-project-filtered
        :args (s/cat :this some?
                     :project-id ::commons-spec/id
                     :filter-fn fn?)
        :ret (s/coll-of (s/keys :req-un [::commons-spec/id ::commons-spec/name ::analysistime])))
(defn all-analyses-in-project-filtered
  "Return all analyses in project with filter option"
  [this project-id filter-fn]
  (-all-analyses-in-project-filtered this project-id filter-fn))

(s/fdef analysis-by-id
        :args (s/cat :this some?
                     :project-id ::commons-spec/id
                     :analysis-id ::commons-spec/id)
        :ret (s/nilable (s/keys :opt-un [::commons-spec/id ::dir ::commons-spec/name])))
(defn analysis-by-id
  "Return the analysis data by id"
  [this project-id analysis-id]
  (-analysis-by-id this project-id analysis-id))
