(ns codescene.features.code-coverage.parsers.ncover-parser
  (:require [codescene.features.code-coverage.parser :refer [Parser]]
            [codescene.features.code-coverage.parsers.xml-utils :as xml-utils]
            [codescene.features.code-coverage.parsers.utils :as utils]
            [clojure.java.io :as io]
            [clojure.data.xml :as xml]
            [clojure.string :as str]
            [medley.core :as m]))

(def ^:private supported-metrics [:line-coverage :sequence-point-coverage])

(defn- parse-path-and-counts 
  [parse-options
   [path {:keys [lines-total lines-covered lines-uncovered seqp-total seqp-covered]}]]
  (let [forward-slashed-path (str/replace path "\\" "/")]
    (m/assoc-some {:path                    forward-slashed-path
                   :name                    (-> forward-slashed-path (str/split #"/") last)}
                  :line-coverage (merge
                                  (utils/->entry (count lines-covered) lines-total)
                                  (when (utils/needs-detailed-line-coverage? parse-options)
                                    {:details {:covered lines-covered
                                               :uncovered lines-uncovered}}))
                  :sequence-point-coverage (utils/->entry seqp-covered seqp-total))))

(defn- seqpnt-covered?
  [{{:keys [visitcount]} :attrs :as _seqpnt-el}]
  (some-> visitcount Integer/parseInt pos?))

(defn- seqpnt-uncovered?
  [{{:keys [visitcount]} :attrs :as _seqpnt-el}]
  (some-> visitcount Integer/parseInt zero?))

(defn- seqpnt->line [{:keys [attrs] :as _seqpnt-el}]
  (:line attrs))

(defn- seqpnt->document [{:keys [attrs] :as _seqpnt-el}]
  (:document attrs))

(defn- lines-matching 
  [covered-uncovered-fn? seqpnts]
  (->> seqpnts 
       (filter covered-uncovered-fn?) 
       (map seqpnt->line)
       set))

(defn- counts
  [seqpnt-els]
  (let [covered-seq-pnts (filter seqpnt-covered? seqpnt-els)]
    {:lines-total (->> seqpnt-els (map seqpnt->line) distinct count)
     :lines-covered   (lines-matching seqpnt-covered? seqpnt-els)
     :lines-uncovered (lines-matching seqpnt-uncovered? seqpnt-els)
     :seqp-total (count seqpnt-els)
     :seqp-covered (count covered-seq-pnts)}))

(defn- ->counts-lookup [seqpnts]
  (->> seqpnts
       (group-by seqpnt->document)
       (m/map-vals counts)))

(defn- read-coverage* [reader parse-options]
  (let [xml (-> reader (xml/parse :nntamespace-aware false :support-dtd false))
        seqpnts (xml-utils/sub-nodes-rec :seqpnt xml)
        path->counts (->counts-lookup seqpnts)]
    (map (partial parse-path-and-counts parse-options) path->counts)))

(defn- read-coverage [f parse-options]
  (with-open [r (io/reader f)]
    (doall (read-coverage* r parse-options))))

(defn ->Parser []
  (reify Parser
    (-read-coverage [this reader parse-options] (read-coverage* reader parse-options))
    (-supported-metrics [this] supported-metrics)
    (-id [this] "ncover")
    (-name [this] "NCover")))

(comment
  (def f "./test/codescene/features/code_coverage/testdata/CSharp/NCover1.5.8.xml")
  (def parse-options {:cli-command "check"})
  (read-coverage f parse-options)
  )