(ns codescene.features.code-coverage.parsers.jacoco-parser
  "DTD: https://github.com/jacoco/jacoco/blob/master/org.jacoco.report/src/org/jacoco/report/xml/report.dtd"
  (:require [codescene.features.code-coverage.parser :refer [Parser]]
            [codescene.features.code-coverage.parsers.xml-utils :as xml-utils]
            [codescene.features.code-coverage.parsers.utils :as utils]
            [clojure.java.io :as io]
            [clojure.data.xml :as xml]
            [medley.core :as m]
            [clojure.string :as str]))

(def ^:private supported-metrics [:line-coverage
                                  :branch-coverage])

(defn- extract-counter-attribute
  [el type attribute]
  (some
    (fn [{:keys [attrs]}]
     (when (= type (:type attrs))
       (get attrs attribute)))
    (xml-utils/sub-nodes :counter el)))

(defn safe-parse-int
  "Parse a string into an int, allowing for nil"
  [str]
  (if (nil? str)
    nil
    (Integer/parseInt str)))

(defn- coverage-data [el type]
  (let [covered (-> el (extract-counter-attribute type :covered) safe-parse-int)
        missed (-> el (extract-counter-attribute type :missed) safe-parse-int)]
     (when (not-any? nil? [covered missed])
       (utils/->entry covered (+ covered missed)))))

(defn- line-details-from
  [source-el]
  (let [ci-at (fn [{:keys [attrs]}]
                (safe-parse-int (:ci attrs)))
        mi-at (fn [{:keys [attrs]}]
                (safe-parse-int (:mi attrs)))
        line-number-of (fn [{:keys [attrs]}]
                         (safe-parse-int (:nr attrs)))]
    (->> source-el
         :content
         (filter (comp (partial = :line) :tag))
         (map (fn [a-line]
                {:number (line-number-of a-line)
                 :ci (ci-at a-line)
                 :mi (mi-at a-line)})))))

(defn- executable-lines-in
  [lines]
  (->> lines
       (filter (fn [parsed-line] ; we could have incomplete data where a field is missing
                 (->> parsed-line
                      vals
                      (every? some?))))
       (remove (fn [{:keys [ci mi]}]
                 (and (zero? ci)
                      (zero? mi))))))

(defn- line-numbers-matching
  [coverage-predicate-fn executable-lines]
  (->> executable-lines
       (filter coverage-predicate-fn)
       (map :number)
       set))

(defn- covered-and-uncovered-lines-from
  "Parses info on covered/uncovered lines from the :line element.
   The input format looks like this:
    
    #xml/element{:tag :line, :attrs {:nr 3, :mi 3, :ci 0, :mb 0, :cb 0}}
   
    A line is covered if:
     - ci > 0 ('ci' means 'covered instructions')
   
    A line is uncoved if:
     - ci = 0 _and_ mi > 0 ('mi' means 'missed insructions')
   
   _IF_ both ci and mi are zero, then that means an unexecutable line -> skip it in the report."
  [source-el]
  (let [lines (line-details-from source-el)
        executable (executable-lines-in lines) 
        covered-line?   (fn [a-line] (> (:ci a-line) 0))
        uncovered-line? (fn [{:keys [ci mi]}]
                          (and (zero? ci)
                               (> mi 0)))]
    {:covered (line-numbers-matching   covered-line?   executable)
     :uncovered (line-numbers-matching uncovered-line? executable)}))

(defn- line-coverage
  [source-el parse-options]
  (let [c (coverage-data source-el "LINE")]
    (if (utils/needs-detailed-line-coverage? parse-options)
      (assoc c :details (covered-and-uncovered-lines-from source-el))
      c)))

(defn- branch-coverage [el]
  (coverage-data el "BRANCH"))

(defn- parent-path
  "Returns the path consisting of names of the parent elements (presumably 'group' elems)
  joined by '/'.
  Returns nil if no parent's are given
  or a path ending with `/` signaling a folder."
  [parents {:keys [cli?] :or {cli? false}}]
  (if cli?
    ""  ;when running in cli we need to ignore groups as will be autogenerated by the supath generation
    (->> parents
       (filter #(= :group (:tag %)))
       ;; append the slash here to produce trailing slash
       (map #(-> % :attrs :name (str "/")))
       ;; the %s is added to be able to inject the supath later in between parent and code coevareg file path
       (#(when (seq %) (concat % ["%s"])))
       str/join)))

(defn- parse-source-el 
  [package-name parents 
   {:keys [attrs] :as el}
   parse-options]
  ;; TODO: extract coverage from "COMPLEXITY", "METHOD" and "CLASS" too ?
  (let [file-name (:name attrs)]
    (m/assoc-some {:path (format "%s%s/%s" (parent-path parents parse-options) package-name file-name)
                   :name file-name}
                  :branch-coverage (branch-coverage el)
                  :line-coverage (line-coverage el parse-options))))

(defn- parse-package-el 
  [parse-options
   {:keys [attrs parents] :as el}]
  (let [package-name (:name attrs)
        source-file-els (xml-utils/sub-nodes :sourcefile el)]
    (map (fn [source-file] (parse-source-el package-name parents source-file parse-options))
         source-file-els)))

(defn- read-coverage* [reader parse-options]
  (let [xml (-> reader (xml/parse :namespace-aware false :support-dtd false))
        ;; recursively search for 'package' elements because they can be nested inside 'group'
        package-els (xml-utils/sub-nodes-rec :package xml)]
    (mapcat (partial parse-package-el parse-options) package-els)))

(defn- read-coverage [f parse-options]
  (with-open [r (io/reader f)]
    (doall (read-coverage* r parse-options))))

(defn ->Parser []
  (reify Parser
    (-read-coverage [this reader parse-options] (read-coverage* reader parse-options))
    (-supported-metrics [this] supported-metrics)
    (-id [this] "jacoco")
    (-name [this] "JaCoCo")))

(comment
  (def f "./test/codescene/features/code_coverage/testdata/Java/JaCoCo0.8.3.xml")
  (def f "/Users/jumar/workspace/CODESCENE/CODE/codescene/features/test/codescene/features/code_coverage/testdata/Java/JaCoCo0.8.3_groups.xml")
  (->> (read-coverage f))

  ;;
  )

