(ns codescene.features.code-coverage.parsers.cobertura-parser
  (:require [codescene.features.code-coverage.parser :refer [Parser]]
            [codescene.features.code-coverage.parsers.xml-utils :as xml-utils]
            [codescene.features.code-coverage.parsers.utils :as utils]
            [codescene.features.util.file :as file-util]
            [clojure.java.io :as io]
            [clojure.data.xml :as xml]
            [clojure.set :as set]
            [clojure.string :as str]
            [medley.core :as m]))

(def ^:private supported-metrics [:line-coverage])

(defn- parse-path-and-lines
  [parse-options
   [path {:keys [covered uncovered total]}]]
  (let [forward-slashed-path (str/replace path "\\" "/")
        line-coverage (utils/->entry (count covered) (count total))]
    (m/assoc-some {:path          forward-slashed-path
                   :name          (-> forward-slashed-path (str/split #"/") last)}
                  :line-coverage (merge line-coverage
                                        (when (utils/needs-detailed-line-coverage? parse-options)
                                          {:details {:covered   covered
                                                     :uncovered uncovered}})))))

(defn- line-numbers [line-els]
  (->> line-els
       (map :attrs)
       (map :number)))

(defn- merge-lines 
  "Merge sets of covered/uncovered line nbrs into accumulated map indexed by filename"
  [acc {:keys [attrs] :as class-el}]
  (let [filename (:filename attrs)
        lines (xml-utils/sub-nodes-in class-el [:lines :line])
        lines-matching (fn [covered-uncovered-fn? all-lines]
                         (->> all-lines
                              (filter #(covered-uncovered-fn? (-> % :attrs :hits)))
                              line-numbers
                              set))
        covered-line?   (partial not= "0")
        uncovered-line? (partial = "0")
        total (line-numbers lines)]
    (update acc filename #(merge-with set/union % {:covered   (lines-matching covered-line?   lines)
                                                   :uncovered (lines-matching uncovered-line? lines)
                                                   :total (set total)}))))

(defn- ->lines-lookup [class-locs]
  (reduce merge-lines {} class-locs))

(defn- read-coverage* [reader parse-options]
  (let [xml (-> reader (xml/parse :namespace-aware false :support-dtd false))
        with-trailing-slash (fn [s] (when s
                                      (if (str/ends-with? s "/") s (str s "/"))))
        source (->> (xml-utils/sub-nodes-in xml [:sources :source])
                    (map (comp first :content))
                    (filter not-empty)
                    first
                    with-trailing-slash)
        with-source #(if (-> % (str/replace "\\" "/") file-util/is-simple-file-name) (str source %) %)
        ;;we prefix the filename when is relative path with first not empty source (ensuring trailing slash)
        update-filename (fn [nodes] (map #(update-in % [:attrs :filename] with-source) nodes))
        classes (-> xml (xml-utils/sub-nodes-in [:packages :package :classes :class]) update-filename )
        path->lines (->lines-lookup classes)]
    (map (partial parse-path-and-lines parse-options) path->lines)))

(defn- read-coverage [f parse-options]
  (with-open [r (io/reader f)]
    (doall (read-coverage* r parse-options))))

(defn ->Parser []
  (reify Parser
    (-read-coverage [this reader parse-options] (read-coverage* reader parse-options))
    (-supported-metrics [this] supported-metrics)
    (-id [this] "cobertura")
    (-name [this] "Cobertura")))

(comment
  (def f "./test/codescene/features/code_coverage/testdata/Java/Cobertura2.1.1.xml")
  (def parse-options {:cli-command "check"})
  (read-coverage f parse-options)
)