(ns codescene.features.chat.service.tools
  "Tool execution for chat service.
   
   Handles executing MCP tools and formatting results for the conversation."
  (:require [codescene.features.chat.service.mcp :as mcp]
            [taoensso.timbre :as log]))

;; Maximum characters for tool results sent to LLM.
;; Large responses can cause LLM timeouts or exceed context limits.
(def ^:private max-result-chars 50000)

(defn- truncate-result
  "Truncates result string if it exceeds max-result-chars.
   Keeps the beginning and end of the content for context."
  [result-str]
  (if (<= (count result-str) max-result-chars)
    result-str
    (let [keep-chars (- max-result-chars 200) ; Reserve space for truncation message
          head-chars (int (* keep-chars 0.7))  ; Keep 70% from start
          tail-chars (int (* keep-chars 0.3))  ; Keep 30% from end
          head (subs result-str 0 head-chars)
          tail (subs result-str (- (count result-str) tail-chars))
          omitted (- (count result-str) head-chars tail-chars)]
      (log/warnf "Truncating tool result from %d to %d chars (omitting %d chars from middle)"
                 (count result-str) max-result-chars omitted)
      (str head 
           "\n\n... [" omitted " characters omitted - result truncated for LLM context limits] ...\n\n"
           tail))))

(defn- format-result [tool-result]
  (if (:success tool-result)
    (let [result-str (str "Tool result:\n" (:result tool-result))
          truncated (truncate-result result-str)]
      (log/debugf "Tool result length: %d chars%s" 
                  (count result-str)
                  (if (< (count truncated) (count result-str)) " (truncated)" ""))
      truncated)
    (str "Tool error: " (:error tool-result))))

(defn execute-and-format
  "Executes a tool call and formats the result as a string."
  [ctx tool-call]
  (let [{:keys [auth-token instance-url mcp-config servers]} ctx
        tool-result (mcp/execute-tool (cond-> {:auth-token auth-token
                                               :instance-url instance-url
                                               :mcp-config mcp-config
                                               :tool-name (:name tool-call)
                                               :arguments (:arguments tool-call)}
                                        servers (assoc :servers servers)))]
    (format-result tool-result)))

(defn append-tool-messages
  "Appends assistant message and tool result to conversation."
  [ctx messages content tool-call]
  (log/infof "LLM requested tool: %s" (:name tool-call))
  (let [result-text (execute-and-format ctx tool-call)]
    (conj messages
          {:role :assistant :content content}
          {:role :user :content result-text})))
