(ns codescene.features.chat.service.mcp
  "MCP server registry. Aggregates tools and prompts from multiple MCP servers."
  (:require [codescene.features.chat.mcp-servers.insights :as insights-mcp]
            [codescene.features.chat.service.mcp.server :as server]
            [taoensso.timbre :as log]))

(defn default-servers
  "Returns the default list of MCP server instances."
  []
  [(insights-mcp/create-server)])

(defn- get-server-config
  "Extracts server-specific config from mcp-config using the server's name."
  [mcp-server mcp-config]
  (get mcp-config (server/server-name mcp-server)))

(defn- build-server-context
  "Builds the context for a specific server, including its config."
  [mcp-server context]
  (let [server-config (get-server-config mcp-server (:mcp-config context))]
    (assoc context :config server-config)))

(defn reset-mcp-session!
  "Resets the MCP session for the given auth-token."
  [auth-token]
  (insights-mcp/reset-session! auth-token))

(defn- extract-items-from-map
  "Extracts items from a structured map result (tools or prompts)."
  [result]
  (cond
    (:tools result) (:tools result)
    (:prompts result) (:prompts result)
    :else []))

(defn- extract-errors-from-map
  "Extracts errors from a structured map result."
  [result]
  (get result :errors []))

(defn- normalize-server-result
  "Normalizes a server result to {:items [...] :errors [...]} format.
   Handles both legacy vector and structured map formats."
  [result]
  (if (map? result)
    {:items (extract-items-from-map result)
     :errors (extract-errors-from-map result)}
    {:items result
     :errors []}))

(defn- collect-items-from-server
  "Collects items from a single server and adds them to the accumulator."
  [context list-fn {:keys [items errors]} mcp-server]
  (let [server-context (build-server-context mcp-server context)
        result (list-fn mcp-server server-context)
        normalized (normalize-server-result result)]
    {:items (into items (:items normalized))
     :errors (into errors (:errors normalized))}))

(defn- aggregate-from-servers
  "Aggregates results from all servers using the given list-fn.
   list-fn is called with (list-fn server context) and should return a collection
   or a structured map with :items/:tools/:prompts and :errors.
   Returns {:items [...] :errors [...]}."
  [context list-fn]
  (let [servers (or (:servers context) (default-servers))]
    (reduce (partial collect-items-from-server context list-fn) 
            {:items [] :errors []} 
            servers)))

(defn- extract-items
  "Extracts items from a server response, handling both vector and structured map formats."
  [result]
  (if (map? result)
    (or (:tools result) (:prompts result) [])
    result))

(defn- server-provides-item?
  "Returns the server if it provides an item with the given name, nil otherwise."
  [item-name context list-fn mcp-server]
  (let [server-context (build-server-context mcp-server context)
        result (list-fn mcp-server server-context)
        items (extract-items result)]
    (when (some #(= item-name (:name %)) items)
      mcp-server)))

(defn- find-server-for
  "Finds the MCP server that provides an item with the given name.
   list-fn is called with (list-fn server context) and should return a collection of items with :name keys."
  [item-name context list-fn]
  (let [servers (or (:servers context) (default-servers))]
    (some (partial server-provides-item? item-name context list-fn) servers)))

(defn- execute-on-server
  "Executes an operation on the appropriate MCP server."
  [{:keys [item-name item-type find-fn execute-fn]} context]
  (if-let [mcp-server (find-fn item-name context)]
    (let [server-context (build-server-context mcp-server context)]
      (execute-fn mcp-server server-context))
    (do
      (log/warnf "No MCP server found for %s: %s" item-type item-name)
      {:success false :error (str "Unknown " item-type ": " item-name)})))

(defn list-tools
  "Lists available tools from all registered MCP servers.
   Optionally takes a :servers key in context to override default servers.
   Returns {:tools [...] :errors [...]}."
  [context]
  (let [{:keys [items errors]} (aggregate-from-servers context server/list-tools)]
    {:tools items :errors errors}))

(defn list-prompts
  "Lists available prompts from all registered MCP servers.
   Optionally takes a :servers key in context to override default servers.
   Returns {:prompts [...] :errors [...]}."
  [context]
  (let [{:keys [items errors]} (aggregate-from-servers context server/list-prompts)]
    {:prompts items :errors errors}))

(defn- find-server-for-tool
  "Finds the MCP server that provides the given tool."
  [tool-name context]
  (find-server-for tool-name context server/list-tools))

(defn- find-server-for-prompt
  "Finds the MCP server that provides the given prompt."
  [prompt-name context]
  (find-server-for prompt-name context server/list-prompts))

(defn execute-tool
  "Executes a tool via the appropriate MCP server.
   Optionally takes a :servers key in context to override default servers."
  [{:keys [tool-name] :as context}]
  (execute-on-server {:item-name tool-name
                      :item-type "tool"
                      :find-fn find-server-for-tool
                      :execute-fn server/execute-tool}
                     context))

(defn get-prompt
  "Gets a prompt via the appropriate MCP server.
   Optionally takes a :servers key in context to override default servers."
  [{:keys [prompt-name] :as context}]
  (execute-on-server {:item-name prompt-name
                      :item-type "prompt"
                      :find-fn find-server-for-prompt
                      :execute-fn server/get-prompt}
                     context))
