(ns codescene.features.chat.service.loop
  "Chat loop implementation for chat service.
   
   Handles the recursive conversation loop with tool execution."
  (:require [codescene.features.chat.service.llm :as llm]
            [codescene.features.chat.service.prompt.parser :as parser]
            [codescene.features.chat.service.tools :as tools]))

(def ^:private max-tool-iterations 5)

(defn- build-final-response
  "Builds the successful response map with the parsed content, usage stats, and iteration count."
  [parsed usage iteration]
  {:success true
   :response {:role :assistant :content (:response parsed)}
   :thinking (:thinking parsed)
   :usage usage
   :iterations (inc iteration)})

(defn- build-error-response
  "Builds an error response map with the error message and accumulated usage stats."
  [error usage]
  {:success false :error error :usage usage})

(defn- handle-tool-call
  "Executes the requested tool, appends results to messages, and continues the loop."
  [{:keys [iterate-fn messages iteration] :as ctx} {:keys [parsed content]}]
  (let [tool-call (:tool-call parsed)
        new-messages (tools/append-tool-messages ctx messages content tool-call)]
    (iterate-fn (assoc ctx :messages new-messages :iteration (inc iteration)))))

(defn- handle-parsed-response
  "Routes to tool execution if a tool call is present, otherwise builds the final response."
  [ctx {:keys [parsed] :as response}]
  (if (:tool-call parsed)
    (handle-tool-call ctx response)
    (build-final-response parsed (:usage ctx) (:iteration ctx))))

(defn- handle-llm-success
  "Processes a successful LLM response by parsing it and handling tool calls or final response."
  [ctx result]
  (let [content (:content result)
        new-usage (merge-with (fnil + 0 0) (:usage ctx) (or (:usage result) {}))
        parsed (parser/parse-llm-response content)]
    (if (:success parsed)
      (handle-parsed-response (assoc ctx :usage new-usage) {:parsed parsed :content content})
      (build-error-response "Failed to parse LLM response" new-usage))))

(defn- chat-iteration
  "Single iteration of the chat loop. Passes itself to handlers for recursive tool execution."
  [{:keys [llm-config messages usage iteration llm-fn] :as ctx}]
  (let [call-llm (or llm-fn llm/call-llm)]
    (cond
      (>= iteration max-tool-iterations)
      (build-error-response "Maximum tool iterations reached" usage)

      :else
      (let [result (call-llm llm-config messages)]
        (if (:success result)
          (handle-llm-success ctx result)
          (build-error-response (:error result) usage))))))

(defn run
  "Runs the chat loop with the given context."
  [llm-config context]
  (-> (merge context {:llm-config llm-config
                      :iterate-fn chat-iteration
                      :usage {:prompt-tokens 0 :completion-tokens 0}
                      :iteration 0})
      chat-iteration))
