(ns codescene.features.chat.service.llm
  "LLM provider routing for chat service.
   
   Supports: OpenAI, Anthropic, Azure, Gemini, and custom OpenAI-compatible endpoints."
  (:require [codescene.features.chat.service.llm.anthropic :as anthropic]
            [codescene.features.chat.service.llm.gemini :as gemini]
            [codescene.features.chat.service.llm.openai :as openai]
            [codescene.features.chat.service.llm.provider :as provider]))

(def provider-configs
  "Default configurations for supported LLM providers.
   Each config specifies base-url and chat-endpoint for the provider's API."
  {:openai {:base-url "https://api.openai.com/v1"
            :chat-endpoint "/chat/completions"}
   :anthropic {:base-url "https://api.anthropic.com/v1"
               :chat-endpoint "/messages"}
   :azure {:chat-endpoint "/chat/completions"}
   :gemini {:base-url "https://generativelanguage.googleapis.com/v1beta"
            :chat-endpoint "/models/{model}:generateContent"}
   :custom {:chat-endpoint "/chat/completions"}})

(defn- get-provider
  "Returns the appropriate LLMProvider instance for the given provider type."
  [provider-type]
  (case provider-type
    :anthropic (anthropic/->AnthropicProvider)
    :gemini (gemini/->GeminiProvider)
    (openai/->OpenAIProvider)))

(defn- build-provider-config
  "Builds the full provider config by merging defaults with user overrides.
   Adds api-key and model, uses custom base-url if provided."
  [{:keys [provider model api-key base-url]}]
  (let [defaults (get provider-configs provider (:custom provider-configs))]
    (cond-> (assoc defaults :api-key api-key :model model)
      base-url (assoc :base-url base-url))))

(defn call-llm
  "Calls the configured LLM provider with messages."
  [llm-config messages]
  (let [llm-provider (get-provider (:provider llm-config))
        config (build-provider-config llm-config)]
    (provider/call llm-provider config messages)))
