(ns codescene.features.api.teams
  (:require [codescene.features.util.api :as api-utils]
            [codescene.features.api.core :as api-core]
            [codescene.features.components.developer-settings :as developer-settings]
            [medley.core :as m]))

(defn- with-ref
  [api-prefix developer-settings-id team]
  (assoc team :ref (format "%s%s/teams/%s" (api-utils/developer-settings-url-prefix api-prefix) developer-settings-id (:id team))))

(defn list-result
  [system developer-settings-id {:keys [page filter-str]}]
  (let [teams (developer-settings/all-teams (api-core/api-developer-settings system) developer-settings-id)
        api-prefix (api-core/api-root system)]
    (api-utils/paginated-list-result teams {:page page
                                            :filter-str filter-str
                                            :list-kw :teams
                                            :processing-fn (partial with-ref api-prefix developer-settings-id)})))

(defn post-new-team
  [system developer-settings-id {:keys [team_name]}]
  (let [developer-settings-component (api-core/api-developer-settings system)
        team-exists? (->> (developer-settings/all-teams developer-settings-component developer-settings-id)
                          (some #(= (:name %) team_name)))]
    (if team-exists?
      (api-utils/bad-request (format "team '%s' already exists!" team_name))
      (do
        (developer-settings/add-team developer-settings-component developer-settings-id {:name team_name})
        (api-utils/created {:ok (format "team-created: '%s'" team_name)} {})))))

(defn update-team
  [system developer-settings-id team-id {:keys [team_name]}]
  (let [developer-settings-component (api-core/api-developer-settings system)
        row-updated (developer-settings/update-team developer-settings-component developer-settings-id {:id team-id :name team_name})]
    (if (= 0 row-updated)
      (api-utils/bad-request (format "There's no team with id: %d" team-id))
      (api-utils/created {:ok (format "team-updated: '%s'" team_name)} {}))))

(defn delete-team
  [system developer-settings-id team-id]
  (let [developer-settings-component (api-core/api-developer-settings system)
        team? (->> (developer-settings/all-teams developer-settings-component developer-settings-id)
                  (m/find-first #(= (:id %) team-id)))
        [success reason] (if team?
                           (developer-settings/delete-team developer-settings-component developer-settings-id team-id)
                           [false (format "There's no team with id: %d" team-id)])]
    (if success
      (api-utils/ok {:ok "team-deleted"})
      (api-utils/bad-request reason))))