(ns codescene.features.api.spec.analysis
  (:require [spec-tools.spec :as spec]
            [clojure.spec.alpha :as s]
            [codescene.specs :as specs]
            [codescene.features.api.spec.common :as common-spec]))

(s/def ::id spec/int?)
(s/def ::project_id spec/int?)
(s/def ::project_owner spec/string?)
(s/def ::repository_paths_on_disk (s/coll-of spec/string?))
(s/def ::configured_git_remote_urls (s/coll-of spec/string?))
(s/def ::analysis_destination spec/string?)
(s/def ::dir spec/string?)
(s/def ::repo spec/string?)
(s/def ::revision spec/string?)
(s/def ::analysis_repo_revisions (s/coll-of (s/keys :req-un [::repo ::revision])))
(s/def ::readable_analysis_time spec/string?)
(s/def ::path spec/string?)
(s/def ::number_of_defects spec/int?)
(s/def ::last_modification_age_in_months (s/or :last_modification_age_in_months #{"-"} :last_modification_age_in_months spec/int?))
(s/def ::lines_of_code ::common-spec/optional-decimal-number)
(s/def ::ownership_percentage spec/number?)
(s/def ::knowledge_loss_percentage (s/or :knowledge_loss_percentage #{"-"} :knowledge_loss_percentage spec/number?))
(s/def ::team_fragmentation_percentage spec/number?)
(s/def ::owner spec/string?)
(s/def ::language spec/string?)
(s/def ::change_frequency spec/int?)
(s/def ::cost spec/number?)
(s/def ::primary_team spec/string?)
(s/def ::team_ownership ::common-spec/optional-decimal-number)
(s/def ::number_of_authors spec/number?)
(s/def ::number_of_teams spec/number?)

(s/def ::author_contributions_fragmentation ::common-spec/optional-decimal-number)
(s/def ::teams_contributions_fragmentation ::common-spec/optional-decimal-number)

(s/def ::project (s/keys :req-un [::id ::common-spec/name]
                         :opt-un [::common-spec/description ::project_owner ::analysis_destination
                                  ::repository_paths_on_disk ::configured_git_remote_urls]))

(s/def ::date_type (s/and string? #(not (empty? %)) #(re-matches common-spec/YYYY-MM-DD %)))

(s/def ::page spec/int?)
(s/def ::max_pages spec/int?)
(defmacro paginated [spec & opt-un]
  `(s/keys :req-un [::page ::max_pages ~spec]
           :opt-un ~(vec opt-un)))

(s/def ::date ::date_type)
(s/def ::ref spec/string?)
(s/def ::analysis_status spec/string?)
(s/def ::description spec/string?)
(s/def ::analysis_errors (s/coll-of (s/keys :req-un [::description])) )
(s/def ::projects (s/coll-of (s/keys :req-un [::id ::common-spec/name ::ref]
                                     :opt-un [::analysis_status ::analysis_errors])))
(s/def ::project_list (paginated ::projects))
(s/def ::username ::specs/non-empty-string)
(s/def ::password ::specs/non-empty-string)
(s/def ::role_name (s/or :role_name nil? :role_name ::specs/non-empty-string))
(s/def ::user (s/keys :req-un [::id ::username ::ref ::role_name]))
(s/def ::users (s/coll-of ::user))
(s/def ::users_list (paginated ::users))

(s/def ::role_id spec/int?)
(s/def ::role (s/keys :req-un [::id ::common-spec/name ::ref]))
(s/def ::roles (s/coll-of ::role))
(s/def ::roles_list (paginated ::roles))
(s/def ::default_role (s/keys :req-un [::id ::common-spec/name]))
(s/def ::default_role_result (s/keys :req-un [::default_role]))

(s/def ::type ::specs/non-empty-string)
(s/def ::provider_name ::specs/non-empty-string)
(s/def ::active? boolean?)
(s/def ::codescene_default_role spec/int?)
(s/def ::host string?)
(s/def ::port string?)
(s/def ::connection_timeout string?)
(s/def ::response_timeout string?)
(s/def ::bind_dn_format string?)
(s/def ::search_base string?)
(s/def ::ssl? boolean?)
(s/def ::teams_url string?)
(s/def ::teamname_field string?)
(s/def ::access_token_url string?)
(s/def ::scope string?)
(s/def ::authorization_url string?)
(s/def ::username_field string?)
(s/def ::client_id string?)
(s/def ::user_url string?)
(s/def ::client_secret string?)

(def auth_provider_config_spec
  {:ldap {:spec (s/keys :req-un [::host ::port ::connection_timeout ::response_timeout
                                 ::bind_dn_format ::search_base ::ssl?])
          :message (str "For the 'ldap' authentication provider config you need to provide the 'host', 'port', 'connection_timeout', 'response_timeout',"
                        "'bind_dn_format', 'search_base' and 'ssl?'.")}
   :oauth2 {:spec (s/keys :req-un [::teams_url ::teamname_field ::access_token_url ::scope ::provider_name
                                   ::authorization_url ::username_field ::client_id ::user_url ::client_secret])
            :message (str "For the 'oauth2' authentication provider config you need to provide the 'teams_url', 'teamname_field', 'access_token_url', 'scope',"
                          "'provider_name', 'authorization_url', 'username_field', 'client_id', 'user_url' and 'client_secret'.")}})
(s/def ::auth_provider_config map?)
(s/def ::authentication_provider (s/keys :req-un [::id ::type ::active? ::codescene_default_role ::auth_provider_config]))
(s/def ::authentication_providers (s/coll-of (s/keys :req-un [::id ::type ::provider_name ::ref])))
(s/def ::authentication_providers_list (paginated ::authentication_providers))

(s/def ::identifier ::specs/non-empty-string)
(s/def ::roles_mapping (s/keys :req-un [::identifier ::role]))
(s/def ::roles_mapping_list (s/coll-of ::roles_mapping))

(s/def ::developer_setting (s/keys :req-un [::id ::ref ::common-spec/name]))
(s/def ::developer_settings (s/coll-of ::developer_setting))
(s/def ::developer_settings_list (paginated ::developer_settings))

(s/def ::team_name ::specs/non-empty-string)
(s/def ::team (s/keys :req-un [::id ::ref ::common-spec/name]))
(s/def ::teams (s/coll-of ::team))
(s/def ::teams_list (paginated ::teams))

(s/def ::exclude_from_all_analyses spec/boolean?)
(s/def ::former_contributor spec/boolean?)
(s/def ::team_ref spec/string?)
(s/def ::email (s/nilable spec/string?))
(s/def ::emails (s/coll-of spec/string?))
(s/def ::team_id spec/int?)
(s/def ::developer (s/keys :req-un [::id ::common-spec/name ::team_name ::exclude_from_all_analyses ::former_contributor ::ref]
                           :opt-un [::team_ref ::email ::emails]))
(s/def ::developers (s/coll-of ::developer))
(s/def ::developers_list (paginated ::developers))

(s/def ::group_id spec/int?)
(s/def ::group (s/keys :req-un [::id ::ref ::common-spec/name]))
(s/def ::groups (s/coll-of ::group))
(s/def ::groups_list (paginated ::groups))

(s/def ::code-health-score (s/or :missing-score #{"-"} :present-score spec/number?))

(s/def ::current_score ::code-health-score)
(s/def ::month_score ::code-health-score)
(s/def ::year_score ::code-health-score)
(s/def ::system_mastery (s/or :system_mastery #{"-"} :system_mastery spec/number?))
(s/def ::active_developers (s/or :active_developers #{"-"} :active_developers spec/number?))
(s/def ::statistic spec/string?)
(s/def ::value spec/number?)
(s/def ::language spec/string?)
(s/def ::number_of_files spec/any?)
(s/def ::blank spec/any?)
(s/def ::comment spec/any?)
(s/def ::code spec/any?)
(s/def ::title spec/string?)
(s/def ::severity spec/string?)
(s/def ::commits spec/number?)
(s/def ::entities spec/number?)
(s/def ::change_entities spec/number?)
(s/def ::authors_count spec/number?)
(s/def ::active_authors_count spec/number?)

(s/def ::hotspots_code_health_now_weighted_average ::code-health-score)
(s/def ::hotspots_code_health_month_weighted_average ::code-health-score)
(s/def ::hotspots_code_health_year_weighted_average ::code-health-score)

(s/def ::code_health_weighted_average_current ::code-health-score)
(s/def ::code_health_weighted_average_last_month ::code-health-score)
(s/def ::code_health_weighted_average_last_year ::code-health-score)

(s/def ::code_health_now_worst_performer ::code-health-score)
(s/def ::code_health_month_worst_performer ::code-health-score)
(s/def ::code_health_year_worst_performer ::code-health-score)

(s/def ::code_health_now_weighted_average ::code-health-score)

(s/def ::summary (s/keys :opt-un [::commits ::entities ::change_entities ::authors_count ::active_authors_count]))
(s/def ::file_summary (s/coll-of (s/keys :opt-un [::language ::number_of_files ::blank ::comment ::code])))

(s/def ::high_level_metrics (s/keys :req-un [::current_score ::month_score ::year_score ::lines_of_code
                                             ::active_developers ::system_mastery]
                                    :opt-un [::hotspots_code_health_now_weighted_average
                                             ::hotspots_code_health_month_weighted_average
                                             ::hotspots_code_health_year_weighted_average

                                             ::code_health_weighted_average_current
                                             ::code_health_weighted_average_last_month
                                             ::code_health_weighted_average_last_year

                                             ::code_health_now_worst_performer
                                             ::code_health_month_worst_performer
                                             ::code_health_year_worst_performer]))

(s/def ::overall_coverage spec/number?)
(s/def ::hotspot_coverage spec/number?)
(s/def ::uncovered_files spec/number?)
(s/def ::uncovered_lines spec/number?)
(s/def ::metric string?)

(s/def ::metrics (s/coll-of (s/keys :req-un [::overall_coverage
                                             ::hotspot_coverage
                                             ::uncovered_files
                                             ::uncovered_lines
                                             ::metric])))

(s/def ::code_coverage (s/keys :req-un [::overall_coverage
                                        ::hotspot_coverage
                                        ::uncovered_files
                                        ::uncovered_lines
                                        ::metrics]))

(s/def ::analysis (s/keys :req-un [::id ::common-spec/name ::project_id ::dir ::analysis_repo_revisions ::readable_analysis_time
                                   ::common-spec/description ::high_level_metrics ::summary ::file_summary]
                          :opt-un [::code_coverage]))

(s/def :codescene.features.api.spec.analysis.commit-list/rev spec/string?)
(s/def :codescene.features.api.spec.analysis.commit-list/date spec/string?)
(s/def :codescene.features.api.spec.analysis.commit-list/author spec/string?)
(s/def :codescene.features.api.spec.analysis.commit-list/files spec/int?)
(s/def :codescene.features.api.spec.analysis.commit-list/loc_added spec/int?)
(s/def :codescene.features.api.spec.analysis.commit-list/loc_deleted spec/int?)
(s/def :codescene.features.api.spec.analysis.commit-list/repository spec/string?)
(s/def :codescene.features.api.spec.analysis.commit-list/ticket_id spec/string?)

(s/def ::commit-object (s/keys :req-un [:codescene.features.api.spec.analysis.commit-list/rev
                                        :codescene.features.api.spec.analysis.commit-list/date
                                        :codescene.features.api.spec.analysis.commit-list/author
                                        :codescene.features.api.spec.analysis.commit-list/files
                                        :codescene.features.api.spec.analysis.commit-list/loc_added
                                        :codescene.features.api.spec.analysis.commit-list/loc_deleted
                                        :codescene.features.api.spec.analysis.commit-list/repository]
                               :opt-un [:codescene.features.api.spec.analysis.commit-list/ticket_id]))

(s/def :codescene.features.api.spec.analysis.commit-list/commits (s/coll-of ::commit-object))

(s/def ::paginated-commit-list (paginated :codescene.features.api.spec.analysis.commit-list/commits))

(s/def :codescene.features.api.spec.analysis.issue-list/date spec/string?)
(s/def :codescene.features.api.spec.analysis.issue-list/id spec/string?)
(s/def :codescene.features.api.spec.analysis.issue-list/loc_added spec/int?)
(s/def :codescene.features.api.spec.analysis.issue-list/loc_deleted spec/int?)
(s/def :codescene.features.api.spec.analysis.issue-list/files spec/int?)
(s/def :codescene.features.api.spec.analysis.issue-list/authors spec/int?)
(s/def :codescene.features.api.spec.analysis.issue-list/first_commit spec/string?)
(s/def :codescene.features.api.spec.analysis.issue-list/last_commit spec/string?)
(s/def :codescene.features.api.spec.analysis.issue-list/closed_at spec/string?)
(s/def :codescene.features.api.spec.analysis.issue-list/status spec/string?)
(s/def :codescene.features.api.spec.analysis.issue-list/cycle_time_hours spec/int?)
(s/def :codescene.features.api.spec.analysis.issue-list/href spec/string?)

(s/def :codescene.features.api.spec.analysis.issue-list.file_change/file spec/string?)
(s/def :codescene.features.api.spec.analysis.issue-list.file_change/loc_added spec/int?)
(s/def :codescene.features.api.spec.analysis.issue-list.file_change/loc_deleted spec/int?)

(s/def :codescene.features.api.spec.analysis.issue-list/file_change
  (s/keys :req-un [:codescene.features.api.spec.analysis.issue-list.file_change/file
                   :codescene.features.api.spec.analysis.issue-list.file_change/loc_added
                   :codescene.features.api.spec.analysis.issue-list.file_change/loc_deleted]))

(s/def :codescene.features.api.spec.analysis.issue-list/file_changes
  (s/coll-of :codescene.features.api.spec.analysis.issue-list/file_change))
(s/def ::issue-object (s/keys :req-un
                              [:codescene.features.api.spec.analysis.issue-list/id]
                              :opt-un
                              [:codescene.features.api.spec.analysis.issue-list/date
                               :codescene.features.api.spec.analysis.issue-list/loc_added
                               :codescene.features.api.spec.analysis.issue-list/loc_deleted
                               :codescene.features.api.spec.analysis.issue-list/files
                               :codescene.features.api.spec.analysis.issue-list/file_changes
                               :codescene.features.api.spec.analysis.issue-list/authors
                               :codescene.features.api.spec.analysis.issue-list/first_commit
                               :codescene.features.api.spec.analysis.issue-list/last_commit
                               :codescene.features.api.spec.analysis.issue-list/closed_at
                               :codescene.features.api.spec.analysis.issue-list/status
                               :codescene.features.api.spec.analysis.issue-list/cycle_time_hours
                               :codescene.features.api.spec.analysis.issue-list/href]))

(s/def :codescene.features.api.spec.analysis.issue-list/issues (s/coll-of ::issue-object))

(s/def ::paginated-issue-list (paginated :codescene.features.api.spec.analysis.issue-list/issues))

(s/def ::analyses (s/coll-of (s/keys :req-un [::id ::common-spec/name ::ref])))
(s/def ::analyses_list (paginated ::analyses ;; req-un
                         ;; opt-un:
                                  ::date))

(s/def ::authors (s/coll-of spec/string?))
(s/def ::current_total_lines_of_code spec/number?)
(s/def ::churned_lines_of_code_by_current_contributors spec/number?)
(s/def ::churned_lines_of_code_by_former_contributors spec/number?)

(s/def ::codebase_experience_for_current_contributors spec/number?)
(s/def ::languages_list (s/coll-of (s/keys :req-un [::language ::authors ::current_total_lines_of_code
                                                    ::churned_lines_of_code_by_current_contributors
                                                    ::churned_lines_of_code_by_former_contributors
                                                    ::codebase_experience_for_current_contributors])))

;; Code Health
;;
(s/def ::code_health_rule spec/string?)
(s/def ::code_health_rule_description spec/string?)
(s/def ::code_health_rules (s/coll-of (s/keys :req-un [::code_health_rule
                                                       ::code_health_rule_description])))
(s/def ::code_health_rules_list (s/keys :req-un [::code_health_rules]))

(s/def ::author spec/string?)
(s/def ::relative_total_contribution_percentage spec/number?)
(s/def ::relative_contribution_percentage spec/number?)
(s/def ::languages (s/coll-of (s/keys :req-un [::language ::relative_contribution_percentage ::relative_total_contribution_percentage])))
(s/def ::language_detail (s/coll-of (s/keys :req-un [::author ::relative_total_contribution_percentage ::languages])))

(s/def ::code_health (s/nilable (s/keys :req-un [::year_score ::current_score ::month_score])))

(s/def ::system_health (s/nilable (s/keys :opt-un [::year_score ::current_score ::month_score ::code_health_now_worst_performer ::code_health_now_weighted_average])))
(s/def ::recommendation (s/nilable (s/keys :req-un [::title ::severity])))
(s/def ::recommendations (s/coll-of ::recommendation))
(s/def ::count spec/integer?)
(s/def ::code_smell spec/string?)
(s/def ::rule_set spec/string?)
(s/def ::code_health_rule_violation (s/nilable (s/keys :req-un [::code_smell ::rule_set]
                                                       :opt-un [::count])))
(s/def ::code_health_rule_violations (s/coll-of ::code_health_rule_violation))
(s/def ::category spec/string?)
(s/def ::text spec/string?)
(s/def ::goal (s/nilable (s/keys :req-un [::category ::text])))
(s/def ::goals (s/coll-of ::goal))

(s/def ::file (s/keys :opt-un [::path ::last_modification_age_in_months ::common-spec/name ::lines_of_code ::ownership_percentage
                               ::knowledge_loss_percentage ::owner ::language ::change_frequency ::number_of_defects
                               ::cost ::code_health ::primary_team ::team_ownership ::number_of_authors
                               ::number_of_teams
                               ::recommendations
                               ::code_health_rule_violations
                               ::goals
                               ::author_contributions_fragmentation
                               ::teams_contributions_fragmentation]))

(s/def ::files (s/coll-of ::file))
(s/def ::files_list (paginated ::files))

;; This affords both types as we've had string on some props for a time
;; It displays in Swagger as boolean
(s/def ::boolean-string (s/or :b boolean? :s #{"true" "false"}))
(s/def ::order_by #{"lines_of_code" "change_frequency" "number_of_defects" "code_health" "cost"})
(s/def ::pr_provider_types #{"github" "github-app" "bitbucket" "bitbucket-server" "gitlab" "azure" "gerrit" "plugin-gerrit"})
(s/def ::pr_provider_types_new #{"github" "bitbucket" "bitbucket-server" "gitlab" "azure" "gerrit" "plugin-gerrit"})
(s/def ::pm_provider_types #{"azure" "github" #_"shortcut" "jira" "trello" "youtrack" "gitlab"})

(s/def ::components (s/coll-of (s/keys :req-un [::common-spec/name ::ref])))

(s/def ::components_list (paginated ::components))

(s/def ::component (s/keys :opt-un [::last_modification_age_in_months ::common-spec/name ::lines_of_code ::ownership_percentage
                                    ::knowledge_loss_percentage ::owner ::language ::change_frequency ::number_of_defects
                                    ::cost ::system_health ::team_fragmentation_percentage ::system_mastery
                                    ::primary_team
                                    ::number_of_authors
                                    ::number_of_teams
                                    ::team_ownership
                                    ::code_coverage]))

(s/def ::urls (s/coll-of spec/string?))
(s/def ::url spec/string?)
(s/def ::path spec/string?)
(s/def ::branch (s/nilable spec/string?))
(s/def ::repository (s/keys :req-un [::url] :opt-un [::branch]))
(s/def ::repositories (s/coll-of ::repository))
(s/def ::local-path (s/keys :req-un [::path] :opt-un [::branch]))
(s/def ::local-paths (s/coll-of ::local-path))
(s/def ::generate-architectural-component spec/boolean?)
(s/def ::local-paths (s/coll-of spec/string?))

;; Commit activity trend
;;
(s/def ::authors_at_date ::authors_count)
(s/def ::revisions_at_date ::change_frequency)
(s/def ::commit_activity_trend (s/coll-of (s/keys :req-un [::date
                                                           ::authors_at_date
                                                           ::revisions_at_date])))
(s/def ::commit_activity_trend_response (s/keys :req-un [::commit_activity_trend]))

(s/def ::months_contributing spec/number?)
(s/def ::last_contribution spec/string?)
(s/def ::first_contribution spec/string?)
(s/def ::primary_component_owner spec/number?)
(s/def ::commit_pattern spec/string?)
(s/def ::author spec/string?)
(s/def ::primary_file_owner spec/number?)
(s/def ::lines_of_code_removed spec/number?)
(s/def ::lines_of_code_added spec/number?)
(s/def ::former_contributor spec/boolean?)
(s/def ::lines_of_code_net spec/number?)
(s/def ::author_statistics_response (s/coll-of (s/keys :req-un [::months_contributing
                                                                ::last_contribution
                                                                ::first_contribution
                                                                ::primary_component_owner
                                                                ::commit_pattern
                                                                ::author
                                                                ::primary_file_owner
                                                                ::lines_of_code_removed
                                                                ::lines_of_code_added
                                                                ::commits
                                                                ::former_contributor
                                                                ::lines_of_code_net])))

(s/def ::first_commit_date (s/nilable spec/string?))
(s/def ::risk_description (s/nilable spec/string?))
(s/def ::repository_name (s/nilable spec/string?))
(s/def ::lead_time_first_commit_in_hours spec/number?)
(s/def ::delivery_risk spec/number?)
(s/def ::lead_time_to_merge_in_hours spec/number?)
(s/def ::merged spec/boolean?)
(s/def ::branch_duration_in_hours spec/number?)
(s/def ::last_commit_date (s/nilable spec/string?))
(s/def ::contributing_authors spec/number?)
(s/def ::branch_statistics_response (s/coll-of (s/keys :req-un [::first_commit_date
                                                                ::risk_description
                                                                ::repository_name
                                                                ::lead_time_first_commit_in_hours
                                                                ::delivery_risk
                                                                ::lead_time_to_merge_in_hours
                                                                ::merged
                                                                ::branch_duration_in_hours
                                                                ::last_commit_date
                                                                ::commits
                                                                ::branch
                                                                ::contributing_authors])))

(s/def ::file_name spec/string?)
(s/def ::code_health_score spec/number?)
(s/def ::loc spec/number?)
(s/def ::revisions spec/number?)
(s/def ::friction spec/number?)
(s/def ::friction_last_month spec/number?)
(s/def ::friction_last_year spec/number?)
(s/def ::target_type spec/string?)
(s/def ::td-refactoring-target (s/keys :opt-un [::file_name
                                                ::code_health_score
                                                ::loc
                                                ::revisions
                                                ::friction
                                                ::friction_last_month
                                                ::friction_last_year
                                                ::target_type]))
(s/def ::result (s/coll-of ::td-refactoring-target))
(s/def ::paginated-technical-debt-list (paginated ::result))
(s/def ::hotspots (s/coll-of ::td-refactoring-target))
(s/def ::paginated-technical-debt-hotspots-list (paginated ::hotspots))

(s/def ::event-type spec/string?)
(s/def ::event-properties spec/map?)
