(ns codescene.features.api.developers
  (:require
   [clojure.set :as set]
   [codescene.analysis.paths :as paths]
   [codescene.features.util.api :as api-utils]
   [codescene.features.api.core :as api-core]
   [codescene.features.components.developer-settings :as developer-settings]
   [codescene.features.components.analysis :as analysis]
   [codescene.features.util.csv :as csv-util]
   [medley.core :as m]))

(defn- bad-request-for-id
  [developer-id]
  (api-utils/bad-request (format "There's no developer with id: %d" developer-id)))

(defn- processing-for-api
  "add ref and team-ref to developer, rename excluded, exdev and team_id"
  [api-prefix developer-settings-id developer]
  (let [developer-settings-url-prefix (api-utils/developer-settings-url-prefix api-prefix)
        ref (format "%s%s/developers/%s" developer-settings-url-prefix developer-settings-id (:id developer))
        team-ref (format "%s%s/teams/%s" developer-settings-url-prefix developer-settings-id (:teamid developer))]
    (-> (select-keys developer [:id :name :team_name :excluded :exdev :email :emails])
        (update-in [:team_name] #(or % "none"))
        (set/rename-keys {:excluded :exclude_from_all_analyses :exdev :former_contributor})
        (assoc :ref ref)
        (cond-> (:teamid developer) (assoc :team_ref team-ref)))))

(defn- all-authors
  "return a list of authors from the latest analyses of every project using this developer-settings-id"
  [system developer-settings-id]
  (let [analysis-component (api-core/api-analyses system)
        developer-settings-component (api-core/api-developer-settings system)
        project-ids (developer-settings/project-ids-using developer-settings-component developer-settings-id)
        analysis->local-result-path (fn [project-id]
                                      (when-let [analysis-id (:id (analysis/latest-successful-analysis analysis-component project-id))]
                                        (analysis/local-analysis-result-path analysis-component project-id analysis-id)))
        local-analysis-result-paths (->> project-ids
                                         (map #(analysis->local-result-path %))
                                         (filter some?))]
    (->> local-analysis-result-paths
         (map #(csv-util/read-csv (% paths/all-authors-csv)))
         (reduce concat)
         (distinct)
         (map #(set/rename-keys % {:author :name :author-email :email})))))

(defn- ->email-lookup
  [system developer-settings-id]
  (->> (all-authors system developer-settings-id)
       (m/index-by :name)
       (m/map-vals :email)))

(defn- ->emails-lookup
  [system developer-settings-id]
  (->> (all-authors system developer-settings-id)
       (group-by :name)
       (m/map-vals #(mapv :email %))))

(defn- augmented-developers
  "return a list of developers augmented with email taken from the authors list"
  [system developer-settings-id]
  (let [developer-settings-component (api-core/api-developer-settings system)
        name->email (->email-lookup system developer-settings-id)
        name->emails (->emails-lookup system developer-settings-id)
        developers (developer-settings/known-developers developer-settings-component developer-settings-id)]
    (map (fn [{:keys [name] :as dev}]
           (-> dev
                (assoc :emails (get name->emails name []))
                (m/assoc-some :email (name->email name))))
         developers)))

(comment
  (require '[cacsremoteservice.features.system :as features-system])
  (let [system features-system/*system*
        developer-settings-id 20
        developer-settings-component (api-core/api-developer-settings system)
        authors (all-distinct-authors system developer-settings-id)
        developers (developer-settings/known-developers developer-settings-component developer-settings-id)]
  {:authors authors
   :all-authors  (all-authors system developer-settings-id)
   :developers developers})

  (augmented-developers features-system/*system* 20)
  (emails-lookup features-system/*system* 20)
 ,)

(defn list-result
  [system developer-settings-id {:keys [page filter-str]}]
  (let [api-prefix (api-core/api-root system)]
    (api-utils/paginated-list-result (augmented-developers system developer-settings-id)
                                     {:page page
                                      :filter-str filter-str
                                      :list-kw :developers
                                      :processing-fn (partial processing-for-api api-prefix developer-settings-id)})))

(defn- get-developer
  [system developer-settings-id developer-id]
  (->> (developer-settings/known-developers (api-core/api-developer-settings system) developer-settings-id)
       (m/find-first #(= (:id %) developer-id))))

(defn update-developer
  [system developer-settings-id developer-id {:keys [team_id former_contributor exclude_from_all_analyses]}]
  (let [developer-settings-component (api-core/api-developer-settings system)
        team-exists? (->> (developer-settings/all-teams developer-settings-component developer-settings-id)
                          (some #(= (:id %) team_id)))
        developer (get-developer system developer-settings-id developer-id)]
    (cond
      (not developer)
      (bad-request-for-id developer-id)

      (and team_id (not team-exists?))
      (api-utils/bad-request (format "There's no team with id: %d" team_id))

      :else
      (do
        (let [updated-developer (cond-> developer
                                        team_id (assoc :teamid team_id)
                                        (some? former_contributor) (assoc :exdev former_contributor)
                                        (some? exclude_from_all_analyses) (assoc :excluded exclude_from_all_analyses))]
          (developer-settings/update-developer developer-settings-component developer-settings-id updated-developer))
          (api-utils/created {:ok (format "developer-updated: '%s'" (:name developer))} {})))))

(defn delete-developer
  [system developer-settings-id developer-id]
  (if (get-developer system developer-settings-id developer-id)
    (do
      (developer-settings/delete-developer (api-core/api-developer-settings system) developer-settings-id developer-id)
      (api-utils/ok {:ok "developer-deleted"}))
    (bad-request-for-id developer-id)))

(defn delete-developer-team
  [system developer-settings-id developer-id]
  (if-let [developer (get-developer system developer-settings-id developer-id)]
    (do
      (developer-settings/update-developer (api-core/api-developer-settings system) developer-settings-id (assoc developer :teamid nil))
      (api-utils/ok {:ok (format "team-removed from developer: '%s'" (:name developer))}))
    (bad-request-for-id developer-id)))
