(ns codescene.features.api.analyses
  (:require [codescene.features.util.api :as api-utils]
            [codescene.features.api.core :as api-core]
            [codescene.features.components.analysis :as analysis]
            [codescene.features.util.number :as util-number]
            [codescene.presentation.display :as display]
            [clojure.string :as string]
            [clojure.set :as set]
            [codescene.features.analysis.files :as analysis-files]
            [codescene.features.analysis.util :as analysis-util]
            [codescene.features.analysis.util.technical-debt :as analysis-technical-debt-util]
            [codescene.features.util.csv :as csv-util]
            [codescene.analysis.paths :as paths]
            [slingshot.slingshot :refer [try+ throw+]]
            [ring.util.codec :as ring-codec]
            [semantic-csv.core :as sc]
            [taoensso.timbre :as log]))

(defn- get-local-analysis-dir
  "get local analysis dir results path, for cloud checks if results are already on disk and
  if not downloads them from S3 and unarchives the results.
  Returns the absolute path to the results folder"
  [system project-id analysis-id]
  (-> (api-core/api-analyses system)
      (analysis/local-analysis-result-path project-id analysis-id)
      (apply [""])))

(defn analysis-ref
  [api-prefix project-id analysis-id]
  (format "%s%s/analyses/%s" (api-utils/project-url-prefix api-prefix) project-id analysis-id))

(defn- with-ref
  [api-prefix project-id analysis]
  (assoc analysis :ref (analysis-ref api-prefix project-id (:id analysis))))

(defn list-result
  [system project-id {:keys [page filter-str filter-fn map-params]}]
  (let [api-prefix (api-core/api-root system)
        analysis (api-core/api-analyses system)
        all-analyses-in-project (analysis/all-analyses-in-project-filtered analysis project-id filter-fn)]
    ;; we need to return a vector to avoid issue from compojure-api which return reverses LazySeqs
    ;;(see https://github.com/metosin/compojure-api/issues/412)
    (api-utils/paginated-list-result (into [] all-analyses-in-project)
                                     {:page page
                                      :filter-str filter-str
                                      :list-kw :analyses
                                      :processing-fn (partial with-ref api-prefix project-id)
                                      :extra-map map-params})))

(defn- select-analysis-keys
  [analysis]
  (select-keys analysis [:id :name :project-id :dir :analysis-repo-revisions
                         :readable-analysis-time :description :high_level_metrics
                         :summary :file_summary :code_coverage]))

(defn- parsed-health-field-or-default
  [dashboard field]
  (or (display/->maybe-double (get dashboard field)) "-"))

(defn- detailed-code-health-kpis-when-available
  [{:keys [full-code-health-average-now full-code-health-average-last-month full-code-health-average-last-year] :as dashboard}]
  (when (or full-code-health-average-now
            full-code-health-average-last-month
            full-code-health-average-last-year)
    (let [parse (partial parsed-health-field-or-default dashboard)]
      {:hotspots_code_health_now_weighted_average (parse :hotspots-code-health-now-weighted-average)
       :hotspots_code_health_month_weighted_average (parse :hotspots-code-health-month-weighted-average)
       :hotspots_code_health_year_weighted_average (parse :hotspots-code-health-year-weighted-average)

       :code_health_weighted_average_current (parse :full-code-health-average-now)
       :code_health_weighted_average_last_month (parse :full-code-health-average-last-month)
       :code_health_weighted_average_last_year (parse :full-code-health-average-last-year)

       :code_health_now_worst_performer (parse :full-code-health-worst-performer-now)
       :code_health_month_worst_performer (parse :full-code-health-worst-performer-last-month)
       :code_health_year_worst_performer (parse :full-code-health-worst-performer-last-year)})))

(defn with-high-level-metrics-and-summary
  [path-fn analysis]
  (let [dashboard (analysis-util/dashboard-parameters-for path-fn)
        loss (display/->maybe-int (:knowledgelosspercentage dashboard))
        lines_of_code (display/->maybe-int (apply str (filter #(Character/isDigit %) (:codesize dashboard))))
        summary (map (fn [{:keys [statistic value]}]
                       {(-> (string/lower-case statistic)
                            (string/replace #" " "_")
                            keyword) (display/->maybe-int value)})
                     (csv-util/read-csv (path-fn paths/summary-csv)))
        file-summary (csv-util/read-csv (path-fn paths/file-summary-csv))]
    (merge {:summary      (set/rename-keys (into {} summary) {:authors :authors_count :active_authors :active_authors_count})
            :file_summary (map #(api-utils/rename-and-format-to-int % {:files :number_of_files} [:blank :files :comment :code]) file-summary)
            :high_level_metrics
            (merge
              {:current_score     (util-number/floor-number-when-exists (or (:biomarkercurrent dashboard) "-"))
               :month_score       (util-number/floor-number-when-exists (or (:biomarkermonthback dashboard) "-"))
               :year_score        (util-number/floor-number-when-exists (or (:biomarkeryearback dashboard) "-"))
               :active_developers (display/->maybe-int (get dashboard :activeauthors "-"))
               :lines_of_code     (or lines_of_code "-")
               :system_mastery    (if (integer? loss) (- 100 loss) "-")}
              ; NOTE: the old code health scores (:current_score, etc.) are kept for backwards compatibility, but
              ; we also add the new set of scores:
              (detailed-code-health-kpis-when-available dashboard))}
           (analysis-util/code-coverage-summary path-fn)
           analysis)))

(defn analysis-result
  [system project-id analysis-id]
  (try+
   (let [analysis (api-core/api-analyses system)
         analysis-by-id (analysis/analysis-by-id analysis project-id analysis-id)]
     (if analysis-by-id
       (api-utils/ok (-> (analysis/local-analysis-result-path analysis project-id analysis-id)
                         (with-high-level-metrics-and-summary analysis-by-id)
                         select-analysis-keys))
       (api-utils/not-found (format "There's no analysis matching the ID %s for project with ID %s" analysis-id project-id))))
   (catch [:type :invalid-analyses-request] {:keys [message]}
     (api-utils/bad-request message))))

(defn latest-analysis-result
  [system project-id]
  (let [analysis (api-core/api-analyses system)
        latest-analysis (analysis/latest-successful-analysis analysis project-id)]
    (if latest-analysis
      (analysis-result system project-id (:id latest-analysis))
      (api-utils/not-found (format "There's no analysis for project with ID %s" project-id)))))

(defn analysis-files-list-result
  [system project-id analysis-id options]
  (try+
    (if-let [analysis-by-id (-> (api-core/api-analyses system)
                                (analysis/analysis-by-id project-id analysis-id))]
      (let [path-fn (-> (api-core/api-analyses system)
                        (analysis/local-analysis-result-path project-id analysis-id))
            data (analysis-files/file-list (assoc analysis-by-id 
                                                  :dir (path-fn "")
                                                  :path-fn path-fn))]
        (api-utils/ok (analysis-util/paginated-ordered-data data (assoc options :results-k :files))))
      (api-utils/not-found (format "There's no analysis matching the ID %s for project with ID %s" analysis-id project-id)))
    (catch [:type :invalid-analyses-request] {:keys [message]} (api-utils/not-found message))))

(defn with-analysis-by-id
  [system project-id analysis-id payload-fn]
  (let [api-analysis (api-core/api-analyses system)]
    (if (analysis/analysis-by-id api-analysis project-id analysis-id)
      (api-utils/ok (-> (analysis/local-analysis-result-path api-analysis project-id analysis-id)
                        payload-fn))
      (api-utils/not-found (format "There's no analysis matching the ID %s for project with ID %s" analysis-id project-id)))))

(defn with-latest-analysis
  [system project-id payload-fn]
  (if-let [latest-analysis (analysis/latest-successful-analysis (api-core/api-analyses system) project-id)]
    (with-analysis-by-id system project-id (:id latest-analysis) payload-fn)
    (api-utils/not-found (format "There's no analysis for project with ID %s" project-id))))

(defn wrap-csv-not-found [f]
  (fn [req]
    (try+
      (f req)
      (catch [:kind :csv-file-not-found] e
        (log/warn e)
        (api-utils/bad-request "Issue data not found. Is the PM integration enabled?")))))

(defn commits
  [system project-id analysis-id options]
  (with-analysis-by-id system project-id analysis-id (fn get-commits [path-fn]
                                                       (analysis-util/commits options path-fn))))

(defn latest-analysis-commits
  [system project-id options]
  (with-latest-analysis system project-id (fn get-commits [path-fn]
                                            (analysis-util/commits options path-fn))))

(defn issues
  [system project-id analysis-id options]
  (with-analysis-by-id system project-id analysis-id (fn get-issues [path-fn]
                                                       (analysis-util/issues options path-fn))))

(defn latest-analysis-issues
  [system project-id options]
  (with-latest-analysis system project-id (fn get-issues [path-fn]
                                            (analysis-util/issues options path-fn))))

(defn author-statistics
  [system project-id analysis-id]
  (with-analysis-by-id system project-id analysis-id analysis-util/author-statistics))

(defn latest-author-statistics
  [system project-id]
  (with-latest-analysis system project-id analysis-util/author-statistics))

(defn branch-statistics
  [system project-id analysis-id]
  (with-analysis-by-id system project-id analysis-id analysis-util/branch-statistics))

(defn latest-branch-statistics
  [system project-id]
  (with-latest-analysis system project-id analysis-util/branch-statistics))

(defn technical-debt
  [system project-id analysis-id {:keys [refactoring-targets? results-k] :as opts}]
  (->> #(-> (analysis-technical-debt-util/technical-debt % {:refactoring-targets? refactoring-targets?})
            (analysis-util/paginated-ordered-data (assoc opts :results-k (or results-k :result))))
       (with-analysis-by-id system project-id analysis-id)))

(defn latest-technical-debt
  [system project-id {:keys [refactoring-targets? results-k] :as opts}]
  (->> #(-> (analysis-technical-debt-util/technical-debt % {:refactoring-targets? refactoring-targets?})
            (analysis-util/paginated-ordered-data (assoc opts :results-k (or results-k :result))))
       (with-latest-analysis system project-id)))

#_(defn author-statistics
  [system project-id analysis-id]
  (let [api-analysis (api-core/api-analyses system)]
    (if (analysis/analysis-by-id api-analysis project-id analysis-id)
      (api-utils/ok (-> (analysis/local-analysis-result-path api-analysis project-id analysis-id)
                        analysis-util/author-statistics))
      (api-utils/not-found (format "There's no analysis matching the ID %s for project with ID %s" analysis-id project-id)))))

#_(defn latest-author-statistics
  [system project-id]
  (if-let [latest-analysis (analysis/latest-successful-analysis (api-core/api-analyses system) project-id)]
    (author-statistics system project-id (:id latest-analysis))
    (api-utils/not-found (format "There's no analysis for project with ID %s" project-id))))

#_(defn branch-statistics
  [system project-id analysis-id]
  (let [api-analysis (api-core/api-analyses system)]
    (if (analysis/analysis-by-id api-analysis project-id analysis-id)
      (api-utils/ok (-> (analysis/local-analysis-result-path api-analysis project-id analysis-id)
                        analysis-util/branch-statistics))
      (api-utils/not-found (format "There's no analysis matching the ID %s for project with ID %s" analysis-id project-id)))))

#_(defn latest-branch-statistics
  [system project-id]
  (if-let [latest-analysis (analysis/latest-successful-analysis (api-core/api-analyses system) project-id)]
    (branch-statistics system project-id (:id latest-analysis))
    (api-utils/not-found (format "There's no analysis for project with ID %s" project-id))))

(defn latest-analysis-files-list-result
  [system project-id options]
  (if-let [latest-analysis (analysis/latest-successful-analysis (api-core/api-analyses system) project-id)]
    (analysis-files-list-result system project-id (:id latest-analysis) options)
    (api-utils/not-found (format "There's no analysis for project with ID %s" project-id))))

(defn- component-list-names
  "return a sequence of components names from the analysis results"
  [system project-id analysis-id]
  (let [dir (get-local-analysis-dir system project-id analysis-id)]
    (analysis-util/component-list-names-in-analysis dir)))

(defn- component-processing-fn
  [component-name project-id analysis-id]
  (let [ref (->> (ring-codec/url-encode component-name)
                 (format "%s%s/analyses/%s/components/%s"
                         (api-utils/project-url-prefix api-utils/default-api-root)
                         project-id
                         analysis-id))]
    {:name component-name :ref ref}))

(defn components-list-result
  [system project-id analysis-id {:keys [page filter-str] :as _options}]
  (if (-> (api-core/api-analyses system)
          (analysis/analysis-by-id project-id analysis-id))
    (api-utils/paginated-list-result (component-list-names system project-id analysis-id)
                                     {:page          page
                                      :filter-str    filter-str
                                      :list-kw       :components
                                      :processing-fn (fn [c] (component-processing-fn c project-id analysis-id))})
    (api-utils/not-found (format "There's no analysis matching the ID %s for project with ID %s" analysis-id project-id))))

(defn component-result
  [system project-id analysis-id component-name]
    (let [dir (get-local-analysis-dir system project-id analysis-id)
          component (when (not-empty dir) (analysis-util/get-component-by-name dir component-name))]
      (cond
        (empty? dir)
        (api-utils/not-found (format "There's no analysis matching the ID %s for project with ID %s" analysis-id project-id))

        (some? component)
        (api-utils/ok component)

        :else
        (api-utils/not-found (format "There's no component %s for project with ID %d" component-name project-id)))))

(defn latest-components-list-result
  [system project-id options]
  (if-let [latest-analysis (analysis/latest-successful-analysis (api-core/api-analyses system) project-id)]
    (components-list-result system project-id (:id latest-analysis) options)
    (api-utils/not-found (format "There's no analysis for project with ID %s" project-id))))

(defn latest-component-result
  [system project-id component]
  (if-let [latest-analysis (analysis/latest-successful-analysis (api-core/api-analyses system) project-id)]
    (component-result system project-id (:id latest-analysis) component)
    (api-utils/not-found (format "There's no analysis for project with ID %s" project-id))))

(defn get-analysis
  "return analysis by if for the project id with kw :path-fn to generate the local abalysis path"
  [system project-id analysis-id]
  (let [api-analyses (api-core/api-analyses system)
        path-fn (analysis/local-analysis-result-path api-analyses project-id analysis-id)
        analysis (analysis/analysis-by-id api-analyses project-id analysis-id)]
    (if analysis (assoc analysis :path-fn path-fn)
                 {:message (format "There's no analysis matching the ID %s for project with ID %s" analysis-id project-id)})))

(defn components-file-list-result
  "order components files ascending using comparator and return the page of the given size"
  [{:keys [path-fn] :as analysis} component options]
  (try+
    (analysis-util/validate-component analysis component)
    (let [analysis (assoc analysis :dir (path-fn ""))
          data (analysis-files/file-list analysis component)]
      (api-utils/ok (analysis-util/paginated-ordered-data data (assoc options :results-k :files))))
    (catch [:type :invalid-analyses-request] {:keys [message]} (api-utils/not-found message))))

(defn latest-components-file-list-result
  [system project-id component options]
  (let [api-analyses (api-core/api-analyses system)
        latest-analysis (analysis/latest-successful-analysis api-analyses project-id)]
    (if latest-analysis
      (components-file-list-result (->> (analysis/local-analysis-result-path api-analyses project-id (:id latest-analysis))
                                        (assoc latest-analysis :path-fn))
                                   component
                                   options)
      (api-utils/not-found (format "There's no analysis for project with ID %s" project-id)))))

;; Commit trends
;;
(defn commit-activity-trend
  "Responds with the commit activity trend, revisions + authors per week."
  [{:keys [path-fn message]}]
  (try+
    (when message
      (throw+ {:type :invalid-analyses-request :message message}))
    (let [trend (->> (path-fn paths/revision-trend-csv)
                     csv-util/read-csv
                     (sc/cast-with {:authors sc/->int
                                    :revisions sc/->int})
                     (mapv (fn [{:keys [date authors revisions]}]
                             {:date date
                              :authors_at_date authors
                              :revisions_at_date revisions})))]
      (api-utils/ok {:commit_activity_trend trend}))
    (catch [:type :invalid-analyses-request] {:keys [message]} (api-utils/not-found message))))

(defn latest-commit-activity-trend
  "Responds with the commit activity trend, revisions + authors per week."
  [system project-id]
  (let [api-analyses (api-core/api-analyses system)
        latest-analysis (analysis/latest-successful-analysis api-analyses project-id)]
    (if latest-analysis
      (commit-activity-trend (->> (analysis/local-analysis-result-path api-analyses project-id (:id latest-analysis))
                                  (assoc latest-analysis :path-fn)))
      (api-utils/not-found (format "There's no analysis for project with ID %s" project-id)))))
