lexer grammar CppSharedRulesLexer;

// Pre-processor directives: since we already handle top-level blocks we're free to
// ignore multi-line macros, etc. All that we need es to identify conditional compilation
// directives _inside_ functions. We need to parse them away because they may lead to
// unbalanced curly braces otherwise.

SHARP : '#' [ \t]*;

PUBLIC: 'public';
PROTECTED: 'protected';
PRIVATE: 'private';

PRE_PROC_IF : SHARP 'if';
PRE_PROC_IFDEF : SHARP 'ifdef';
PRE_PROC_IFNDEF : SHARP 'ifndef';

PRE_PROC_ELSE : SHARP 'else';
PRE_PROC_ELIF:  SHARP 'elif';

PRE_PROC_ENDIF: SHARP 'endif';

PRE_PROC_DEFINE : SHARP 'define' (~[\\\r\n]*? [\\] '\r'? '\n')+? (~[\\]*? '\r'? '\n') -> skip;

DEFINED: 'defined';

PRE_PROC_SKIP_DEFINE : SHARP 'define' ~('\r' | '\n' | '\\')+?  '\r'? '\n' -> skip;
INCLUDE : SHARP 'include' ~('\r' | '\n')+? '\r'? '\n' -> skip;
PRAGMA : SHARP 'pragma' ~('\r' | '\n')+? '\r'? '\n' -> skip;
UNDEF : SHARP 'undef'  ~('\r' | '\n')+? '\r'? '\n' -> skip;

EXTERN_C : 'extern "C"' ~('{')*? LeftBrace;

fragment ESCAPED : '\\\\' | '\\"';
LITERAL : '"' ( ESCAPED | ~('\n'|'\r') )*? '"';

LITERAL_CHAR : '\'' . '\'';

NOEXCEPT: 'noexcept';
THROW: 'throw';

CLASS : 'class';
STRUCT : 'struct';
NAMESPACE : 'namespace';

// Needed for method arguments
UNSIGNED: 'unsigned';

// Tokens for Google Test
GOOGLE_TEST: 'TEST_F' | 'TEST' | 'TYPED_TEST_P' | 'TYPED_TEST';

// Tokens for MS CppUnitTest framework
TEST_CLASS: 'TEST_CLASS';
TEST_METHOD: 'TEST_METHOD';
TEST_METHOD_INITIALIZE: 'TEST_METHOD_INITIALIZE';
TEST_METHOD_CLEANUP: 'TEST_METHOD_CLEANUP';
TEST_CLASS_INITIALIZE: 'TEST_CLASS_INITIALIZE';
TEST_CLASS_CLEANUP: 'TEST_CLASS_CLEANUP';
TEST_MODULE_INITIALIZE: 'TEST_MODULE_INITIALIZE';
TEST_MODULE_CLEANUP: 'TEST_MODULE_CLEANUP';

// Tokens for MS Minecraft test framework, see https://trello.com/c/cLRclis5/51-support-microsoft-test-framework-for-c-code-health
// Ideally, these would be an extension but I don't know how to extend our code at the ANTLR level.
BR_TEST_CLASS: 'BR_TEST_CLASS';
BR_TEST_CLASS_INITIALIZE: 'BR_TEST_CLASS_INITIALIZE';
BR_TEST_CLASS_CLEANUP: 'BR_TEST_CLASS_CLEANUP';
BR_TEST_METHOD_INITIALIZE: 'BR_TEST_METHOD_INITIALIZE';
BR_TEST_METHOD_CLEANUP: 'BR_TEST_METHOD_CLEANUP';
BR_TEST_METHOD: 'BR_TEST_METHOD';

// Reserved keywords that can "look" as function blocks:
IF: 'if';
FOR: 'for' | 'foreach';
WHILE: 'while';
CONTINUE: 'continue';
GOTO: 'goto';
CASE: 'case';
DEFAULT: 'default';
CATCH: 'catch';
RETURN: 'return';

LeftParen : '(';
RightParen: ')';

LeftBrace : '{';
RightBrace : '}';

COLON : ':';
SEMICOLON : ';';

Whitespace : [ \t]+ -> skip;

BlockComment: '/*' .*? '*/' -> skip;
LineComment: '//' ~[\r\n]* -> skip;

NEWLINE : '\r'? '\n' -> skip;

SCOPER : '::';

ARROW: '->';

// Match 'operator' before ID to avoid the ID rule to lex the keyword.
OPERATOR : 'operator()' | ('operator' ~('(' | ')' | '{' | '}')+);
SCOPED_OPERATOR : ID (SCOPER ID)*? SCOPER OPERATOR;

ID : [a-zA-Z_][a-zA-Z0-9_]*;


SCOPED_NAME : SCOPER? ID (SCOPER ID)+;

DEREFERENCE : ARROW | '.';
DEREFERENCED_OBJECT : (ID | SCOPED_NAME)  (DEREFERENCE (ID | SCOPED_NAME))+;

AMP: '&';
STAR: '*';

INT : [0-9]+;
FLOAT: INT '.' INT;

ANY_CHAR : .; // Put this lexer rule last to give it the lowest precedence
