(ns codescene.features.repository-provider.github.coverage
  (:require [clojure.string :as str]
            [codescene.features.project.pr-integration :as pi]
            [codescene.features.repository-provider.github.api :as api]
            [codescene.features.repository-provider.github.checks :as checks]
            [codescene.features.delta.coverage.pr-check :as pr-check]
            [codescene.features.repository-provider.github.review :as review]
            [taoensso.timbre :as log]))

(defn post-pending-run
  [auth hook-body pull-request]
  (checks/create-check-run auth
                           (->> hook-body :repository :url)
                           {:name (->> pull-request :base :ref (format "CodeScene Code Coverage (%s)"))
                            :head-commit-sha (->> pull-request :head :sha)
                            :external-id (->> pull-request :url (str "COV#"))}))

(defn post-review [authed-client pull-request-url body sha]
  (let [{:keys [output conclusion]} body
        spec (review/review-spec sha
                                 []
                                 "cs-code-coverage"
                                 (constantly (:summary output)))]
    (try
      (review/new-review authed-client
                         pull-request-url
                         spec
                         (= conclusion "success"))
      nil
      (catch Exception e
        (if-let [{:keys [message errors]} (ex-data e)]
          (str "Couldn't post review, provider returned " (str/join ", " (cons message errors)))
          (throw e))))))

(defn update-check-run
  [authed-client-fn metadata body sha]
  (let [{:keys [app-installation-id check-run-url pull-request-url]} metadata
        auth-client (authed-client-fn app-installation-id)]
    (if check-run-url
      (checks/update-check-run auth-client check-run-url body)
      (let [pull-request (api/api-request auth-client :get pull-request-url {})]
        (checks/create-complete-check-run
          auth-client
          (-> pull-request :head :repo :url)
          ;; skip external ID so that our webhook ignores events related to it
          (merge body
                 {:name (->> pull-request :base :ref (format "CodeScene Code Coverage (%s)"))
                  :head-commit-sha sha}))))
      (post-review auth-client pull-request-url body sha)))

(defn result-output
  [coverage-results]
  (let [renderable (pr-check/result->renderable coverage-results)]
    {:conclusion (if (= :fail (pr-check/check-status coverage-results)) "failure" "success")
     :output {:title (pr-check/title renderable :github)
              :summary (checks/trim-body (str (pr-check/summary renderable :github)
                                         "\n\n"
                                         (pr-check/itemized renderable :github)))}}))

(defn post-cov-result [PrIntegrationConfig repo result metadata]
  (log/infof "Posting coverage results to GitHub %s" (pr-check/result-coord result))
  (update-check-run (constantly (pi/-authed-client PrIntegrationConfig repo))
                    metadata
                    (result-output (:coverage-results result))
                    (:commit-sha result)))


(defmethod pr-check/post-cov-result :github
  [PrIntegrationConfig repo result metadata]
  (post-cov-result PrIntegrationConfig repo result metadata))