(ns codescene.features.repository-provider.bitbucket.result
  (:require [codescene.features.delta.file-regions :as file-regions]
            [codescene.features.delta.result.pull-request :as result]
            [codescene.features.repository-provider.providers :as providers]
            [medley.core :as m]))

(defn report-update
  ([state result-url] {:result state :link result-url})
  ([state result-url details] {:result state :link result-url :details details})
  ([state result-url details data annotations]
   {:result state :details details :data data :link result-url :annotations annotations}))

(defn build-update [state result-url text]
  (m/assoc-some {:state state :result-url result-url} :text text))

(defn error-update [error result-url]
  {:comment (str "### CodeScene PR Check\n"
                 (result/render-error-title error)
                 "\n---\n"
                 (result/render-error error))
   :build (build-update "FAILED" result-url (result/render-error-title error))
   :report (report-update "FAILED" result-url (str (result/render-error-title error)
                                                    " --- "
                                                    (result/render-error error)))})

(defn pending-update [result-url]
  {:build (build-update "INPROGRESS" result-url nil)
   :report (report-update "PENDING" result-url)})


(defn skipped-update
  "Close a build with the skipped status and the reason keyword.

  Currently there is no landing page for our delta analysis."
  [provider-ref reason result-url]
  (let [comment-title (result/render-error-title reason)
        sidebar-msg (format "Skipped: %s" comment-title)]
    {:comment (when (or (-> provider-ref :result-options :always-comment?) (providers/negative-skip? reason))
                (format "### CodeScene PR Check Skipped\n%s\n---\n%s"
                        comment-title
                        (result/render-error reason)))
     :build (build-update (if (providers/negative-skip? reason) "FAILED" "SUCCESSFUL") result-url sidebar-msg)
     :report (report-update (if (providers/negative-skip? reason) "FAILED" "PASSED")
                            result-url
                            sidebar-msg)}))

(defn violations-data [title cnt]
  {:title title
   :type "TEXT"
   :value (if (pos-int? cnt)
            (format "%d violations" cnt)
            "-")})

(defn report-data
  [{:keys [filtered-findings] :as pr-presentable}]
  (let [res (filtered-findings
              (fn [{:keys [old-name]} _ {:keys [change-level]}]
                (when (= :warning change-level)
                  {:new-file? (nil? old-name)})))]
    (filterv
      some?
      [{:title "Code Health Quality Gates: "
        :type "TEXT"
        :value (if (result/negative-review? pr-presentable)
                 "Failed" "OK")}
       (violations-data "Code Health: " (count res))
       (violations-data "New Code: " (count (filter :new-file? res)))
       {:title "Full Report: "
        :type "LINK"
        :value {:href (:result-url pr-presentable)
                :text "CodeScene"}}])))


(defn change-item->annotation [{:keys [description change-level title-prefix category file-name category-description
                                       finding-url location]}]
  (let [title (str title-prefix " " category)]
    (m/assoc-some
      {:path file-name
       :title title
       :annotation_type "CODE_SMELL"
       :summary (str " " title ": " description)
       :external_id (str "codescene" (random-uuid))
       :result (if (= :improvement change-level) "PASSED" "FAILED")
       :details (cond-> description
                  category-description (str ". " category-description))
       ;; if severity is defined, its text and
       :severity (when (= change-level :notice) "MEDIUM")
       :link finding-url}
      :line (:start location))))

(defn change-item->server-annotation [{:keys [description change-level title-prefix category file-name category-description
                                              finding-url location] :as x}]
  (let [title (str title-prefix " " category)]
    (m/assoc-some
      {:path file-name
       :message (cond-> (str " " title ": " description)
                  category-description (str ". " category-description))
       :externalId (str "codescene" (random-uuid))
       :severity (case change-level :notice "MEDIUM"
                                   :improvement "LOW"
                                   :warning "HIGH")
       :link finding-url}
      :type (case change-level
              :notice "CODE_SMELL"
              :warning "CODE_SMELL"
              nil)
      :line (:start location))))

(defn result-update
  "Close a build with the results of the delta analysis.

  Currently, there is no landing page for our delta analysis."
  [{:keys [result-url] :as pr-presentable}
   change-item->annotation
   result-options]
  (let [negative-review? (result/negative-review? pr-presentable)
        annotations? (every? result-options [:report? :annotations?])
        locator (result/cached-locator #(comp first (file-regions/change-locator % true)))]
    {:comment (when (result/comment? pr-presentable result-options)
                (result/presentable->comment pr-presentable :bitbucket (not annotations?)))
     :build (build-update (if negative-review? "FAILED" "SUCCESSFUL")
                          result-url
                          (format "Code Health Quality Gates: %s" (if negative-review? "FAILED" "OK")))
     :report (report-update (if negative-review? "FAILED" "PASSED")
                            result-url
                            "View the detailed and specific findings in CodeScene"
                            (report-data pr-presentable)
                            (map change-item->annotation (result/file-comment-items pr-presentable locator)))}))
