(ns codescene.features.repository-provider.azure.api
  "Azure API.

  We are using specifically DevOps tokens, not general AD tokens.
  It is possible to fetch all Accounts (organizations) for a token.

  Note that there are several endpoints in use here, look for api-request calls with absolute URLs.
  Calling api-request with a relative URL is a call to https://app.vssps.visualstudio.com/_apis/.

  Most endpoints then require that an organization is provided, so we cannot easily request all
  repositories user has access to. What's more, there is a slew of IDs of various kinds, e.g. from their
  documentation:

  VSIDs uniquely identity a user or group within an account.
  They do not uniquely identify users or groups in cross-account scenarios.
  They are persistable and can be exchanged for descriptors.

  Descriptors uniquely identify users and groups across all accounts.
  Over time, we will be changing guidance and APIs to support VSIDs only for storage of a unique identifier
  to a user or group. Descriptors will be used to reference the user/group for a running system.
  We do not recommend persisting descriptors because there are several scenarios where they
  can change overtime: editing the user in the AAD backing store and linking/unlinking an account
  will result in the user getting a new descriptor."
  (:require [clj-http.client :as http-client]
            [clj-http.core :as http-core]
            [clojure.set :as set]
            [clojure.string :as str]
            [clojure.spec.alpha :as s]
            [codescene.features.client.api :as api-client]
            [codescene.features.repository-provider.specs :as specs]
            [codescene.features.repository-provider.providers :as providers]
            [codescene.features.util.maps :refer [map-of]]
            [codescene.util.json :as json]
            [medley.core :as m]
            [meta-merge.core :refer [meta-merge]]
            [medley.core :refer [assoc-some]]
            [org.clojars.roklenarcic.paginator :as page]
            [slingshot.slingshot :refer [try+]]
            [taoensso.timbre :as log]
            [codescene.url.url-utils :as url])
  (:import (org.apache.commons.io.input BOMInputStream)
           (org.apache.http.client.config RequestConfig)))

(def api-version "6.1-preview.1")
(def compatibility-api-version
  "Used by both onprem and cloud."
  "5.0")
(def compatibility-preview-api-version
  "Some endpoints require a preview version. Azure Cloud looks very forgiving
  but onprem solutions have been giving errors."
  "5.0-preview.1")

(defn ->external-id [account-name project-id repository-id]
  (str account-name "/" project-id "/" repository-id))

(defn external-id->parts [external-id]
  (let [[account project repo] (str/split external-id #"/")]
    {:account account :project project :repository repo}))

(defn repo-info->parts [repo-info]
  (if-let [external-id (:external-id repo-info)]
    (let [[account project repo] (str/split external-id #"/")]
      {:account account :project project :repository repo})
    (let [[account project] (str/split (:owner-login repo-info) #"/")]
      {:account account :project project :repository (:repo-slug repo-info)})))

(defn repo-info->account-name
  "Works with repo-info or DB repository shape"
  [repo]
  (-> repo :external-id external-id->parts :account))

(defn- process-exception
  [exception {:keys [url] :as params}]
  (let [{:keys [body status]} (ex-data exception)
        ;; Azure sometimes puts zero-width non-breaking space in JSON which is illegal
        parsed-body (some-> body (str/replace \ufeff \newline) api-client/json-safe-parse)
        message (:message parsed-body (ex-message exception))
        subtype (get parsed-body :typeKey)
        data (map-of message subtype url)]
    (log/debugf "Azure call returned %s %s" status body)
    (log/info exception "Azure call exception" (api-client/printable-req params))
    (case status
      401 (if (some-> message (str/includes? "is not authorized to access this resource"))
            (api-client/ex-forbidden :azure exception data)
            (api-client/ex-unauthorized :azure exception (assoc data :token-expired? true)))
      403 (api-client/ex-forbidden :azure exception data)
      404 (api-client/ex-forbidden :azure exception data)
      (api-client/ex-http-error :azure exception (or status 0) data))))

(defn no-access-ex [body url]
  (let [body-obj (json/parse-string body)
        msg (format "The user is not authorized to access %s. Check if Third Party Application Access is enabled."
                    url)
        body-str (-> body-obj (assoc :message msg) json/generate-string)]
    (api-client/ex-forbidden :azure nil {:body body-str :url url})))

(defn assert-no-redirect-to-login!
  [resp req]
  (when (or (= 302 (:status resp)) (get-in resp [:headers "www-authenticate"]))
    (let [body (slurp (:body resp))]
      (if (str/includes? body "TF400813")
        (throw (no-access-ex body (:url req)))
        (throw (api-client/ex-unauthorized
                 :azure
                 {:message "Redirected to sign-in (Bad credentials)"
                  :url req
                  :body body}))))))

(defn with-unhandled-redirects
  "Since we're reusing the HttpClient instance between calls and also provider services, that unfortunately
  means that we can only have a single redirect strategy. If you specify redirect-strategy and http-client in req
  map, the strategy goes ignored. Strategy is only used when constructing a new client. We will disable redirect
  handling instead, which can be done on per-request basis by providing request config."
  [req]
  (let [rq (-> (http-core/request-config req)
               RequestConfig/copy
               (.setRedirectsEnabled false)
               (.build))]
    (assoc req :http-request-config rq)))

(defn wrap-unauthorized-response
  "Disables redirects and checks if response indicates a redirect to a sign-in page."
  [client]
  (fn
    ([req]
     (doto (client (with-unhandled-redirects req)) (assert-no-redirect-to-login! req)))
    ([req response raise]
     (client (with-unhandled-redirects req)
             (fn [resp]
               (response (doto resp (assert-no-redirect-to-login! req))))
             raise))))


(defn- bom-input-stream [in]
  (when in
    (-> (BOMInputStream/builder) (.setInputStream in) .get)))


(defn wrap-utf-bom
  "Azure sometimes returns UTF-8 BOM, which is ignored (not-removed) by Java's byte->string decode.
  The resulting string has \ufeff at start, which is illegal in JSON."
  [client]
  (fn
    ;; https://commons.apache.org/proper/commons-io/apidocs/org/apache/commons/io/input/BOMInputStream.html
    ([req] (update (client req) :body bom-input-stream))
    ([req response raise]
     (client req (fn [resp] (response (update resp :body bom-input-stream))) raise))))

(s/fdef api-request*
        :args (s/cat :auth-token ::specs/auth-token
                     :method #{:get :post :delete :put :patch :options :head
                               :trace :connect}
                     :url string?
                     :clj-http-params (s/nilable map?)))
(defn api-request*
  "Core function for API requests. Requires ApiClient token.

  Uses augment-request to add some sort of credentials to the request.
  If request returns 401 then on-expired-token will be called on auth-token.

  If that returns a ApiClient, the request will be retried."
  [authed-client method url request-params]
  (let [params (meta-merge
                 {:url url
                  :accept :json
                  :content-type :json
                  :as :auto
                  :request-method method}
                 request-params)
        final-params (-> authed-client
                         (api-client/augment-request params)
                         (api-client/fix-api-url "https://dev.azure.com"))]
    (when-not (:connection-manager final-params)
      (log/warnf "No connection manager, url=%s, avoid for production use." (:url final-params)))
    (try
      (http-client/with-additional-middleware
        [wrap-utf-bom
         wrap-unauthorized-response]
        (http-client/request final-params))
      (catch Exception e
        (-> e
            (process-exception final-params)
            (api-client/retry-token authed-client)
            (api-request* method url request-params))))))

(defn api-request [authed-client method url request-params]
  (:body (api-request* authed-client method url request-params)))

(defn api-page-request
  "A helper function for a request that requests a particular page. It is based on a paging state
  and returns an items+cursor map."
  [authed-client {:keys [cursor add-page]} url request-params]
  (let [params (if cursor
                 (assoc-in request-params [:query-params "continuationToken"] cursor)
                 request-params)
        resp (api-request* authed-client :get url params)]
    (add-page (get-in resp [:body :value])
              (get-in resp [:headers "x-ms-continuationtoken"]))))

(defn api-paginated-load
  [authed-client url request-params]
  (->> #(api-page-request authed-client % url request-params)
       page/paginate-one!
       page/unwrap))

(defn get-pull-request-commits
  "https://docs.microsoft.com/en-us/rest/api/azure/devops/git/pull%20request%20commits/get%20pull%20request%20commits?view=azure-devops-rest-6.1
  requires vso.code"
  ([authed-client pull-request-url]
   (reverse
     (api-paginated-load
       authed-client
       (str pull-request-url "/commits")
       {:query-params {:api-version compatibility-api-version}})))
  ([authed-client repo-info pr-id]
   (let [{:keys [account project repository]} (repo-info->parts repo-info)]
     (get-pull-request-commits authed-client
                               (format "%s/%s/_apis/git/repositories/%s/pullRequests/%s/commits"
                                       account project repository pr-id)))))

(defn get-pull-request-threads
  "https://docs.microsoft.com/en-us/rest/api/azure/devops/git/pull%20request%20threads/list?view=azure-devops-rest-6.1
  requires vso.scope or vso.threads_full"
  [authed-client pr-url]
  (api-paginated-load
    authed-client
    (format "%s/threads" pr-url)
    {:query-params {:api-version compatibility-api-version}}))

(defn add-pull-request-thread
  "https://docs.microsoft.com/en-us/rest/api/azure/devops/git/pull%20request%20threads/create?view=azure-devops-rest-6.1
  requires vso.code_write or vso.threads_full"
  [authed-client pr-url params]
  (api-request
    authed-client
    :post
    (format "%s/threads" pr-url)
    {:query-params {:api-version compatibility-api-version}
     :form-params params}))

(defn delete-pull-request-comment
  "https://docs.microsoft.com/en-us/rest/api/azure/devops/git/pull%20request%20thread%20comments/delete?view=azure-devops-rest-6.0
  requires vso.code_write or vso.threads_full"
  [authed-client thread]
  (api-request
    authed-client
    :delete
    (-> thread :comments first :_links :self :href)
    {:query-params {:api-version compatibility-api-version}}))

(defn get-my-info
  "requires vso.profile
   https://docs.microsoft.com/en-us/rest/api/azure/devops/profile/profiles/get?view=azure-devops-rest-6.1"
  [authed-client]
  (api-request authed-client
               :get
               "https://app.vssps.visualstudio.com/_apis/profile/profiles/me"
               {:query-params {:api-version "6.0"}}))

(defn vsid [{:keys [auth-token] :as authed-client}]
  (or (:vsid auth-token)
      (:id (get-my-info authed-client))))

(defn get-accounts
  "requires vso.profile
  https://docs.microsoft.com/en-us/rest/api/azure/devops/account/accounts/list?view=azure-devops-rest-6.0"
  [authed-client owner-only?]
  (-> (api-request
        authed-client
        :get
        "https://app.vssps.visualstudio.com/_apis/accounts"
        {:query-params (merge {:api-version api-version}
                              (if owner-only? {:ownerId (vsid authed-client)}
                                              {:memberId (vsid authed-client)}))})
      :value))

(defn descriptor->vsid
  [authed-client acc-name descriptor]
  (-> (api-request
        authed-client
        :get
        (format "https://vssps.dev.azure.com/%s/_apis/graph/storagekeys/%s"
                acc-name
                descriptor)
        {:query-params {:api-version api-version}})
      :value))

(defn vsid->descriptor
  [authed-client acc-name vsid]
  (-> (api-request
        authed-client
        :get
        (format "https://vssps.dev.azure.com/%s/_apis/graph/descriptors/%s"
                acc-name
                vsid)
        {:query-params {:api-version api-version}})
      :value))

(defn get-user
  "requires vso.graph
  https://docs.microsoft.com/en-us/rest/api/azure/devops/graph/users/get?view=azure-devops-rest-6.1"
  [authed-client acc-name descriptor]
  (api-request
    authed-client
    :get
    (format "https://vssps.dev.azure.com/%s/_apis/graph/users/%s"
            acc-name
            descriptor)
    {:query-params {:api-version api-version}}))

(defn get-acc-members
  "requires vso.graph
  https://docs.microsoft.com/en-us/rest/api/azure/devops/graph/users/list?view=azure-devops-rest-6.0"
  [authed-client acc-name]
  (api-paginated-load
    authed-client
    (format "https://vssps.dev.azure.com/%s/_apis/graph/users" acc-name)
    {:query-params {:api-version api-version
                    ;; omitted svc which is services
                    :subjectTypes "msa,aad,imp"}}))

;; https://docs.microsoft.com/en-us/azure/devops/organizations/security/namespace-reference?view=azure-devops
(def account-admin-security-ns
  "Security namespace ID for AccountAdminSecurity"
  "11238e09-49f2-40c7-94d0-8f0307204ce4")

(defn account-admin?
  "I've attempted to fetch whether my owner user has any permissions in AccountAdminSecurity
  and it always says no. Fortunately there is a parameter called alwaysAllowAdministrators,
  which will neatly tell us if someone is an administrator by making any permissions call return true,
  on any type of permission."
  [authed-client acc-name]
  (try+
    (->> {:query-params {:api-version api-version
                         :alwaysAllowAdministrators true}}
         (api-request
           authed-client
           :get
           (format "%s/_apis/permissions/%s/%s" acc-name account-admin-security-ns 1)))
    (catch [:type :http-error] _
      nil)))

(defn url-remove-user
  "Remove user from clone URL"
  [azure-url]
  (str/replace azure-url #"//[^@]*@dev.azure.com" "//dev.azure.com"))

(defn ref-remove-prefix
  "Remove prefix for branch heads that Azure uses."
  [branch-ref]
  (str/replace branch-ref "refs/heads/" ""))

(defn into-repository [{:keys [project] :as repository}]
  {:provider-id :azure
   :name (:name repository)
   :clone-url (url-remove-user (:remoteUrl repository))
   :default-branch (or (some-> (:defaultBranch repository) ref-remove-prefix)
                       "master")
   :private? (= "private" (:visibility project))
   :owner-login (str (:account project) "/" (:name project))
   :owner-type "organization"
   :external-id (->external-id (:account project) (:id project) (:id repository))})

(defn get-projects
  "Loads account's projects via a paging state."
  [authed-client private-projects? {:keys [account add-page] :as paging-state}]
  (try+
    (api-page-request authed-client
                      (-> paging-state
                          (api-client/with-item-xf (filter #(or private-projects? (= "public" (:visibility %))))))
                      (format "/%s/_apis/projects" account)
                      {:query-params {:api-version "7.0" :$top 100}})
    (catch [:status 403] _
      (add-page []))))

(defn get-project-repositories
  "Loads project repositories in a paged way."
  [authed-client {:keys [project] :as paging-state}]
  (api-page-request
    authed-client
    (-> paging-state
        (api-client/with-item-xf (map #(dissoc % :account :project))))
    (format "/%s/%s/_apis/git/repositories" (:account project) (:id project))
    {:query-params {:api-version api-version}}))

(defn get-repositories
  "https://docs.microsoft.com/en-us/rest/api/azure/devops/git/repositories/list?view=azure-devops-rest-6.1
  Requires vso.code, if account-name is nil it returns repositories for all user's accounts"
  [authed-client account-name private-projects? max-concurrency]
  (let [get-projects-fn (page/async-fn (partial get-projects authed-client private-projects?) max-concurrency)
        get-repos-fn (page/async-fn (partial get-project-repositories authed-client) get-projects-fn)]
    (->> (if account-name [{:account account-name}]
                          (mapv #(hash-map :account (:accountName %))
                                (get-accounts authed-client false)))
         ;; paginate calls return vector of vectors of maps
         (page/paginate! get-projects-fn {:pages? true})
         (mapcat page/unwrap)
         (map #(hash-map :project %))
         (page/paginate! get-repos-fn {:pages? true})
         (mapcat page/unwrap)
         (mapv into-repository))))

(defn get-repository
  "https://docs.microsoft.com/en-us/rest/api/azure/devops/git/repositories/get%20repository?view=azure-devops-rest-6.1
  Requires vso.code"
  [authed-client repo-info]
  (let [{:keys [account project repository]} (repo-info->parts repo-info)]
    (api-request
      authed-client
      :get
      (format "%s/%s/_apis/git/repositories/%s" account project repository)
      {:query-params {:api-version compatibility-api-version}})))

(defn get-repository-master-branch
  [authed-client repo-info]
  (some-> (get-repository authed-client repo-info)
          :defaultBranch
          providers/branch-ref->branch-name))

(defn repo-accessibility
  [authed-client repo-info]
  (try+
    (let [{:keys [account project]} (repo-info->parts repo-info)
          project (api-request authed-client
                               :get
                               (format "%s/_apis/projects/%s" account project)
                               {:query-params {:api-version api-version}})
          repo (assoc (get-repository authed-client repo-info)
                 ::project (assoc project :account account))]
      (set/rename-keys
        (assoc (into-repository repo)
          :accessible? true)
        {:clone-url :url
         :name :repo-slug}))
    (catch [:type :http-error :status 401] _
      (log/errorf (:throwable &throw-context) "Error getting repository accessibility for %s" (pr-str repo-info))
      repo-info)
    (catch [:type :http-error :status 403] _
      (log/debugf (:throwable &throw-context) "Could not access repository %s. Marking as inaccessible." repo-info)
      (merge repo-info {:private? true :accessible? false}))))

(defn get-repo-branches
  "https://docs.microsoft.com/en-us/rest/api/azure/devops/git/refs/list?view=azure-devops-rest-6.1
  Requires vso.code"
  [authed-client max-results {:keys [add-page no-branch-fallback?] :as paging-state}]
  (if no-branch-fallback?
    (add-page nil)
    (let [{:keys [account project repository]} (repo-info->parts paging-state)
          paging-state (-> paging-state
                           (api-client/with-item-xf (comp (map :name)
                                                          (map providers/branch-ref->branch-name)))
                           (api-client/limit-items max-results))]
      (api-page-request
        authed-client
        paging-state
        (format "/%s/%s/_apis/git/repositories/%s/refs"
                account project repository)
        {:query-params {:api-version api-version :filter "heads/"}}))))

(defn get-repos-branches
  [authed-client repo-infos max-concurrency max-branches]
  (let [get-repo-defaults (page/async-fn (providers/branch-defaults-fn authed-client get-repository-master-branch)
                                         max-concurrency)
        get-branches (partial get-repo-branches authed-client max-branches)]
    (->> repo-infos
         (mapv get-repo-defaults)
         (mapv page/unwrap-async)
         (page/paginate! (page/async-fn get-branches max-concurrency) {})
         (mapv providers/->branch-results))))

(defn- ->hook-info [hook]
  (get-in hook [:_links :self :href]))

(defn get-account-repositories
  "Use this because TFS cannot list repositories by name"
  [authed-client owner project tfs?]
  (let [url (cond
              tfs? (format "/tfs/%s/_apis/git/repositories" owner)
              project (format "/%s/%s/_apis/git/repositories" owner project)
              :else (format "/%s/_apis/git/repositories" owner))]
    (api-paginated-load authed-client url {:query-params {:api-version compatibility-api-version}})))

(defn url->external-id [get-account-repositories url]
  (let [{:keys [project repo owner tfs?] :as parts} (url/azure-url->parts url)
        repositories (get-account-repositories owner project tfs?)
        =ci (fn [s1 s2]
              (if (and s1 s2)
                (.equalsIgnoreCase ^String s1 s2)
                (= s1 s2)))]
    (if-some [r (m/find-first #(and (=ci repo (:name %)) (=ci project (get-in % [:project :name])))
                              repositories)]
      (->external-id owner (get-in r [:project :id]) (:id r))
      (let [msg (format "Failed to find repo and project ids for repo %s" repo)]
        (log/error "Couldn't find repo and project IDs for" parts "in" (pr-str repositories))
        (throw (ex-info msg {}))))))

(defn with-external-ids
  "Get a repo and project ids for a repo name. Throws if repo not found"
  [authed-client repo-infos]
  ;; The API calls need the repo id -> Fetch all repos and find it
  ;; TODO ROK this call is used because TFS doesn't support get single repo by name
  (let [get-repos (memoize (partial get-account-repositories authed-client))]
    (mapv (fn [repo-info]
            (if (:external-id repo-info)
              repo-info
              (assoc repo-info :external-id (url->external-id get-repos (:url repo-info)))))
          repo-infos)))

(defn add-webhook
  "https://docs.microsoft.com/en-us/rest/api/azure/devops/hooks/subscriptions/create?view=azure-devops-rest-6.1
  consumer inputs is {:httpHeaders \"x-token:xxx\", :url \"https://c806135f07c8.ngrok.io/webhooks/azure\"}"
  ([authed-client repo-info callback-url token]
   (let [external-id (-> (with-external-ids authed-client [repo-info]) first :external-id)
         {:keys [account project repository]} (external-id->parts external-id)
         url (if (:tfs? repo-info)
               (format "/tfs/%s/_apis/hooks/subscriptions" account)
               (format "%s/_apis/hooks/subscriptions" account))
         params {:query-params {:api-version compatibility-api-version}
                 :form-params {:publisherId "tfs"
                               :resourceVersion "1.0"
                               :consumerId "webHooks"
                               :consumerActionId "httpRequest"
                               :actionDescription "CodeScene Webhook"
                               :publisherInputs {:repository repository
                                                 :branch ""
                                                 :pullrequestCreatedBy ""
                                                 :pullrequestReviewersContains ""
                                                 :projectId project}
                               :consumerInputs (assoc-some
                                                 {:detailedMessagesToSend "none"
                                                  :messagesToSend "none"
                                                  :url callback-url}
                                                 :httpHeaders (when token (str "x-token:" token)))}}]
     (mapv
       ->hook-info
       [(api-request authed-client :post url (assoc-in params [:form-params :eventType] "git.pullrequest.created"))
        (api-request
          authed-client
          :post
          url
          (-> params
              (assoc-in [:form-params :eventType] "git.pullrequest.updated")
              (assoc-in [:form-params :publisherInputs :notificationType] "PushNotification")))]))))

(defn remove-webhook
  "https://docs.microsoft.com/en-us/rest/api/azure/devops/hooks/subscriptions/delete?view=azure-devops-rest-6.1"
  [authed-client url]
  (api-request
    authed-client
    :delete url
    {:query-params {:api-version compatibility-api-version}}))

(defn get-accounts-webhooks
  "https://docs.microsoft.com/en-us/rest/api/azure/devops/hooks/subscriptions/list?view=azure-devops-rest-6.1"
  [authed-client repo-infos our-hook? concurrency]
  (let [accounts (into #{}
                       (map (fn [{:keys [tfs?] :as repo-info}]
                              {:tfs? tfs? :account (:account (repo-info->parts repo-info))}))
                       repo-infos)
        extract (fn [account {:keys [consumerInputs url modifiedDate status publisherInputs eventType]}]
                  {:external-id (->external-id account (:projectId publisherInputs) (:repository publisherInputs))
                   :url url
                   :callback-url (:url consumerInputs)
                   :secret (some->> consumerInputs :httpHeaders (re-find #"x-token:(.*)") second)
                   :events #{eventType}
                   :enabled? (#{"enabled" "onProbation"} status)
                   :timestamp modifiedDate})
        run-fn (fn [{:keys [account tfs?] :as paging-state}]
                 (let [url (if tfs?
                             (format "/tfs/%s/_apis/hooks/subscriptions" account)
                             (format "%s/_apis/hooks/subscriptions" account))
                       paging-state (api-client/with-item-xf paging-state (comp (map (partial extract account))
                                                                                (filter our-hook?)))]
                   (api-page-request authed-client paging-state url {:query-params {:api-version compatibility-api-version
                                                                                    :consumerActionId "httpRequest"
                                                                                    :consumerId "webHooks"}
                                                                     :socket-timeout 60000})))]
    (page/paginate! (page/async-fn run-fn concurrency) {} accounts)))


(defn add-pull-request-status
  "https://docs.microsoft.com/en-us/rest/api/azure/devops/git/pull%20request%20statuses/get?view=azure-devops-rest-6.1"
  [authed-client pr-url {:keys [status-name status-code description target-url]} azure-status-name]
  (api-request
    authed-client
    :post
    (format "%s/statuses" pr-url)
    {:query-params {:api-version compatibility-preview-api-version}
     :form-params {:state status-code
                   :description (str azure-status-name
                                     " " status-name ": "
                                     description)
                   :targetUrl target-url
                   :context {:genre azure-status-name
                             :name status-name}}}))

(defn host->api-url [host]
  (when (and host (not= host "dev.azure.com"))
    (format "https://%s" host)))
