(ns codescene.features.reports.code-health.code-health-scores
  (:require
   [clojure.math.numeric-tower :as math]
   [codescene.features.presentation.scoring :as scoring]
   [codescene.features.reports.pdf-helpers :as ph]
   [codescene.features.reports.graphics.functions :as graphics]
   [codescene.features.reports.common.pdf-elements :as pdf-elements]
   [codescene.features.reports.common.styling :as styling]
   [medley.core :as m]))


(defn- sub-kpi-value-trimmed [keyname {:keys [sub-kpis]}]
  (let [sub-kpi (m/find-first #(= (:key %) keyname) sub-kpis)
        value (:value sub-kpi)]
    (when value (/ (math/floor (* 10 value)) 10))))

(defn- get-sub-kpis-and-scores [factor-key renderable-month renderable-year]
  (let [current-subkpi (sub-kpi-value-trimmed (name factor-key) (last renderable-month))
        monthago-subkpi (sub-kpi-value-trimmed (name factor-key) (first renderable-month))
        yearago-subkpi (sub-kpi-value-trimmed (name factor-key) (first renderable-year))
        current-score (:score (scoring/score-info-for current-subkpi factor-key))
        monthago-score (:score (scoring/score-info-for monthago-subkpi factor-key))
        yearago-score (:score (scoring/score-info-for yearago-subkpi factor-key))]
    {:current-subkpi  current-subkpi
     :monthago-subkpi monthago-subkpi
     :yearago-subkpi  yearago-subkpi
     :current-score   current-score
     :monthago-score  monthago-score
     :yearago-score   yearago-score}))

(defn- graphics-table [title {:keys [current-subkpi current-score]} img-dir]
  (let [score-color (styling/score->color current-score)
        score-icon (styling/sub-score->icon img-dir current-score)]
    (ph/table
      [:och-nopadding-table] [2 1]
      (ph/row
        (ph/cell
          []
          (ph/image [:c-health-image] score-icon)
          (ph/paragraph [:och-title] title))
        (ph/cell
          []
          (ph/table
            [:och-graphics-table] [35]
            (ph/row
              (ph/cell
                [:och-graphics-cell]
                {:background-layer-fn
                 (ph/draw-using-g2d-fn
                   (graphics/code-health-doughnut-fn {:code-health current-subkpi :fill-color score-color}))}))))))))


(defn- info-table [{:keys [monthago-subkpi yearago-subkpi monthago-score yearago-score]}]
  (let [monthago-score-cell-style (styling/score->cell-style monthago-score)
        monthago-info-text (styling/number->cutoff-string monthago-subkpi)
        yearago-score-cell-style (styling/score->cell-style yearago-score)
        yearago-info-text (styling/number->cutoff-string yearago-subkpi)]
    (ph/table
      [:och-nopadding-table] nil
      (ph/row
        (pdf-elements/info-with-score-color yearago-score-cell-style "1 year ago" yearago-info-text)
        (pdf-elements/info-with-score-color monthago-score-cell-style "1 month ago" monthago-info-text)))))


(defn- score [subkpis-and-scores img-dir title description]
  (ph/cell
    [:och-cell]
    (ph/table
      [:och-table] nil
      (ph/row (ph/cell [] (graphics-table title subkpis-and-scores img-dir)))
      (ph/row (ph/cell [:och-line-cell] (ph/line [:c-line])))
      (ph/row (ph/cell [] (info-table subkpis-and-scores)))
      (ph/row (ph/cell [:och-line-cell] (ph/line [:c-line])))
      (ph/row (ph/cell [] (ph/paragraph [:c-subheader] description))))))


(defn pdf-element [month-trends year-trends {:keys [img-dir]}]
  (let [ch-renderable-month (:code-health month-trends)
        ch-renderable-year (:code-health year-trends)
        h-sub-kpis (get-sub-kpis-and-scores :hotspots-code-health ch-renderable-month ch-renderable-year)
        a-sub-kpis (get-sub-kpis-and-scores :average-code-health ch-renderable-month ch-renderable-year)
        w-sub-kpis (get-sub-kpis-and-scores :worst-performer ch-renderable-month ch-renderable-year)]
    (ph/table [:c-card-content-top-table] nil
              (ph/row (ph/cell [] (ph/paragraph [:c-miniheader] "CURRENT CODE HEALTH SCORE") [:spacer 1]))
              (ph/row
                (ph/cell
                  []
                  (ph/table
                    [] [20 1 20 1 20]
                    (ph/row
                      (score h-sub-kpis img-dir "Hotspot Code Health"
                             "Files with high development activity that are worked on often.")
                      (ph/paragraph [] " ")
                      (score a-sub-kpis img-dir "Average Code Health"
                             "The weighted average health of all the files in the codebase.")
                      (ph/paragraph [] " ")
                      (score w-sub-kpis img-dir "Worst Performer"
                             "The lowest code health score measured in the codebase. "))))))))


